/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.test;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Optional;

import lombok.extern.slf4j.Slf4j;

@Slf4j
public class TestUtils {

  /**
   * Returns the riskscape src dir by searching up from the process's working directory
   */
  public static Optional<Path> getRootProjectPath() {
    Path searchPath = Paths.get(".").toAbsolutePath();
    while (!dirIsRiskscapeSrcRoot(searchPath)) {
      searchPath = searchPath.resolve("..").normalize();

      if (Arrays.asList(File.listRoots()).contains(searchPath.toFile())) {
        return Optional.empty();
      }
    }

    return Optional.of(searchPath);
  }

  /**
   * Simple sniff test to see if the given dir looks like the riskscape root - checks for settings.gradle.  Will
   * throw an exception if the given path is the root of a filesystem
   */
  public static boolean dirIsRiskscapeSrcRoot(Path cwd) {
    File cwdFile = cwd.toFile();

    if (cwdFile.listFiles(fname -> fname.getName().equals("settings.gradle")).length == 1) {
      try {
        return Files.lines(cwd.resolve("settings.gradle"), Charset.defaultCharset()).filter(line ->
          line.trim().equals("rootProject.name = 'riskscape'")).findAny().isPresent();
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    }


    return false;
  }

  /**
   * The API project's classes dir should contain generated i18n resources that need to be loaded specially to work with
   * RiskScape's hierarchical i18n loading system.  The URL returned from this method can be used to construct a
   * `ResourceClassLoader` for use with the `Messages` class's `I18nClassLoader`.
   */
  public static URL getApiClassResourcesURL() {
    Path classesDir = getRootProjectPath().get().resolve("engine/api/build/classes/java/main");
    if (!Files.isDirectory(classesDir)) {
      log.warn("Classes dir is missing: {}", classesDir);
    }

    try {
      return classesDir.toUri().toURL();
    } catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }
  }

}
