/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.resource;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.net.URI;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.RiskscapeIOException;
import nz.org.riskscape.engine.util.Pair;

/**
 * Allows non-mocked testing of a protocol that doesn't look like it's using the file scheme (which often has special
 * behaviour) but is still mapped to the file system.
 */
@RequiredArgsConstructor
public class CustomProtocolTestingResourceLoader implements ResourceLoader {

  private final String scheme;
  private final File basedir;
  private final Map<String, Pair<String, Optional<Path>>> pickledContentMap = new HashMap<>();

  public CustomProtocolTestingResourceLoader(String scheme) {
    this(scheme, null);
  }

  @Override
  public Resource load(URI target) throws ResourceLoadingException {

    Pair<String, Optional<Path>> pickled = pickledContentMap.get(target.getSchemeSpecificPart());

    if (pickled != null) {
      return new Resource() {

        @Override
        public URI getLocation() {
          return target;
        }

        @Override
        public InputStream getContentStream() throws RiskscapeIOException {
          return new ByteArrayInputStream(pickled.getLeft().getBytes());
        }

        @Override
        public String getContentAsString() throws RiskscapeIOException {
          return pickled.getLeft();
        }

        @Override
        public Optional<Path> getLocal() {
          return pickled.getRight();
        }
      };
    }

    if (basedir == null) {
      throw new ResourceLoadingException(target, ResourceProblems.get().notFound(target));
    }

    File file = new File(basedir, target.getSchemeSpecificPart());

    if (!file.exists()) {
      throw new ResourceLoadingException(target, ResourceProblems.get().notFound(target));
    }

    return new Resource() {

      @Override
      public URI getLocation() {
        return target;
      }

      @Override
      public InputStream getContentStream() throws RiskscapeIOException {
        try {
          return new FileInputStream(file);
        } catch (FileNotFoundException e) {
          // shouldn't actually happen - we check it exists earlier
          throw new RiskscapeIOException("File not found", e);
        }
      }
    };
  }

  public void addContent(String schemeSpecificPart, String content) {
    addContent(schemeSpecificPart, content, Optional.empty());
  }

  public void addContent(String schemeSpecificPart, String content, Optional<Path> localPath) {
    pickledContentMap.put(schemeSpecificPart, Pair.of(content, localPath));
  }

  @Override
  public String getId() {
    return scheme;
  }

  @Override
  public String getDescription() {
    return scheme;
  }

  @Override
  public boolean canLoad(URI target) {
    return scheme.equals(target.getScheme());
  }

}
