/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.resource;

import java.io.ByteArrayOutputStream;
import java.io.OutputStream;
import java.net.URI;
import java.util.Base64;

/**
 * A {@link CreateHandle} for use in tests. The content that is written to the handle can be retrieved
 * with {@link #getWritten() }.
 */
public class ByteArrayCreateHandle extends CreateHandle {

  /**
   * Convenience method (for tests) to generate a self contained data URI ((RFC-2397)) for the given string and media
   * type.  Can be loaded back in the engine with the DataURILoader.
   */
  public static URI encodeData(String data, String mediaType) {
    return encodeData(data.getBytes(), mediaType);
  }

  /**
   * Convenience method (for tests) to generate a self contained data URI ((RFC-2397)) for the given bytes and media
   * type.  Can be loaded back in the engine with the DataURILoader.
   */
  public static URI encodeData(byte[] data, String mediaType) {
    String base64 = Base64.getEncoder().encodeToString(data);
    return URI.create("data:" + mediaType + ";base64," + base64);
  }

  ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

  public ByteArrayCreateHandle() {
    super(null);
  }

  @Override
  public OutputStream getOutputStream() throws CreateException {
    return outputStream;
  }

  @Override
  public URI store() throws CreateException {
    String mediaType = "application/octet-stream";
    if (getRequest() != null) {
      mediaType = getRequest().getContentType();
    }

    return encodeData(outputStream.toByteArray(), mediaType);
  }

  /**
   * @return The string that has been written to this handle
   */
  public String getWritten() {
    return outputStream.toString();
  }
}
