/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.pipeline;

import java.net.URI;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.codahale.metrics.MetricRegistry;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.output.ListWriter;
import nz.org.riskscape.engine.output.PipelineJobContext;
import nz.org.riskscape.engine.output.PipelineOutputContainer;
import nz.org.riskscape.engine.output.PipelineOutputOptions;
import nz.org.riskscape.engine.output.PipelineOutputStore;
import nz.org.riskscape.engine.output.SinkParameters;
import nz.org.riskscape.engine.pipeline.sink.ListSink;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * A {@link PipelineJobContext} for use when running a pipeline from a test.
 *
 * Output is saved with {@link ListWriter}s which can be obtained with {@link #getStepToSink() }.
 */
public class TestPipelineJobContext implements PipelineJobContext {

  /**
   * A {@link Sink} that also contains the {@link SinkParameters} used to create it.
   */
  @RequiredArgsConstructor
  public class TestSink extends ListSink {

    public final SinkParameters parameters;

  }

  public class TestOutputContainer implements PipelineOutputContainer {

    public Map<String, Problem> grenades = new HashMap<>();

    @Override
    public ResultOrProblems<Sink> createSinkForStep(SinkParameters parameters) {
      if (grenades.containsKey(parameters.getName())) {
        return ResultOrProblems.failed(grenades.get(parameters.getName()));
      }

      TestSink sink = new TestSink(parameters);
      stepToSink.put(parameters.getName(), sink);
      return ResultOrProblems.of(sink);
    }

    public Map<String, TestSink> getSinkMap() {
      return stepToSink;
    }

    @Override
    public URI getStoredAt() {
      return URI.create("test://localhost");
    }

    @Override
    public Map<String, URI> getPipelineOutputs() {
      // TODO return something meaningful?
      return Collections.emptyMap();
    }

    @Override
    public PipelineOutputStore getStore() {
      return new PipelineOutputStore() {
        @Override
        public String getId() {
          return "test";
        }

        @Override
        public int isApplicable(URI outputLocation) {
          return PipelineOutputStore.PRIORITY_HIGH;
        }

        @Override
        public ResultOrProblems<PipelineOutputContainer> create(URI outputLocation, RealizedPipeline rPipeline,
            PipelineOutputOptions options) {
          return ResultOrProblems.failed(Problems.foundWith("test store does not work"));
        }
      };
    }

    @Override
    public void close() {
    }
  }

  @Getter
  private final RealizedPipeline pipeline;

  @Getter
  private final MetricRegistry progressMetrics = new MetricRegistry();

  @Getter
  private final TestOutputContainer outputContainer = new TestOutputContainer();

  @Getter
  private final Map<String, TestSink> stepToSink = new HashMap<>();

  public TestPipelineJobContext(RealizedPipeline pipeline) {
    this.pipeline = pipeline;
  }

  public TestSink getSink(String stepName) {
    return stepToSink.get(stepName);
  }

}
