/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.gt;

import org.locationtech.jts.geom.Coordinate;

import nz.org.riskscape.engine.SRIDSet;


/**
 * A helper for creating geometries in a NZTM (EPSG:2193) projection.
 *
 * Of note is that this helper will adjust ordinates to be relative to {@link #NZ_ORIGIN_NORTH} and
 * {@link #NZ_ORIGIN_EAST} respectively. This is useful has it allow tests to make geometries with
 * small numbers that would otherwise produce geometries that are not within the EPSG:2193 projected
 * bounds.
 *
 * See also the older {@link NZMGGeometryHelper} for NZMG (EPSG:27200).
 */
public class NZTMGeometryHelper extends BaseGeometryHelper {

  /**
   * These coordinate values form an origin point that is roughly somewhere in the
   * middle of both the NZTM projection and the country (on the outskirts of
   * Levin). We use nice round numbers here to assist with debugging.
   */
  public static final int NZ_ORIGIN_NORTH = 5500000;
  public static final int NZ_ORIGIN_EAST = 1800000;


  public NZTMGeometryHelper(SRIDSet sridSet) {
    super(sridSet, SRIDSet.EPSG2193_NZTM);
  }

  /**
   * Creates a {@link Coordinate} where x and y ordinates are offset from {@link #NZ_ORIGIN_NORTH} and
   * {@link #NZ_ORIGIN_EAST} respectively.
   *
   * This allows test code to use small values for x/y but the returned {@link Coordinate} will be
   * within the EPSG:2193 projected bounds.
   */
  @Override
  public Coordinate toCoordinate(double x, double y) {
    // axis are in the northing/easting or y/x order
    return super.toCoordinate(NZ_ORIGIN_NORTH + y, NZ_ORIGIN_EAST + x);
  }

}
