/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.gt;

import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.io.WKTReader;
import lombok.Getter;
import nz.org.riskscape.engine.CrsHelper;
import nz.org.riskscape.engine.SRIDSet;
import org.geotools.referencing.CRS;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

/**
 *
 * @author glennw
 */
public class GeometryHelper implements CrsHelper {

  @Getter
  private final CoordinateReferenceSystem defaultCRS;
  private final GeometryFactory defaultCRSFactory;
  @Getter
  private final CoordinateReferenceSystem latLongCRS;
  private final WKTReader wktReaderDefaultCRS;
  private final SRIDSet sridSet;

  public GeometryHelper(SRIDSet sridSet) {
    this.sridSet = sridSet;
    try {
      defaultCRS = nzMapGrid();
      defaultCRSFactory = sridSet.getGeometryFactory(defaultCRS);
      latLongCRS = CRS.decode("EPSG:4326");
    } catch (Throwable t) {
      throw new RuntimeException(t);
    }
    wktReaderDefaultCRS = new WKTReader(defaultCRSFactory);
  }

  public Geometry toLatLong(Geometry geom) {
    return sridSet.reproject(geom, sridSet.get(latLongCRS));
  }

  public Geometry fromWKT(String wkt) {
    try {
      return wktReaderDefaultCRS.read(wkt);
    } catch (Throwable t) {
      throw new RuntimeException(t);
    }
  }

  public Point point(int x, int y) {
    return defaultCRSFactory.createPoint(new Coordinate(x, y));
  }

  public Polygon box(int x1, int y1, int x2, int y2) {
    return defaultCRSFactory.createPolygon(new Coordinate[]{
      new Coordinate(x1, y1),
      new Coordinate(x2, y1),
      new Coordinate(x2, y2),
      new Coordinate(x1, y2),
      new Coordinate(x1, y1)
    });
  }

  public LineString line(int x1, int y1, int x2, int y2) {
    return defaultCRSFactory.createLineString(new Coordinate[]{
      new Coordinate(x1, y1),
      new Coordinate(x2, y2)
    });
  }

}
