/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.gt;

import org.geotools.geometry.jts.ReferencedEnvelope;

import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.CoordinateSequence;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.io.WKTReader;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import nz.org.riskscape.engine.CrsHelper;
import nz.org.riskscape.engine.SRIDSet;

import lombok.Getter;

/**
 * A geometry helper that is able to create geometries for a single CRS.
 */
public class BaseGeometryHelper implements CrsHelper {

  protected final SRIDSet sridSet;

  @Getter
  private final CoordinateReferenceSystem crs;

  private final GeometryFactory geometryFactory;

  private final WKTReader wktReader;

  public BaseGeometryHelper(SRIDSet sridSet, String epsgCode) {
    this(sridSet, SRIDSet.epsgToCrs(epsgCode));
  }

  public BaseGeometryHelper(SRIDSet sridSet, CoordinateReferenceSystem crs) {
    this.sridSet = sridSet;
    this.crs = crs;
    try {
      this.geometryFactory = sridSet.getGeometryFactory(crs);
      this.wktReader = new WKTReader(geometryFactory);
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * @return geom reprojected to the CRS from {@link #getCrs() } if necessary.
   */
  public Geometry reproject(Geometry geom) {
    return sridSet.reproject(geom, sridSet.get(getCrs()));
  }

  /**
   * @return envelope reprojected to the CRS from {@link #getCrs() } if necessary.
   */
  public ReferencedEnvelope reproject(ReferencedEnvelope envelope) {
    try {
      return envelope.transform(getCrs(), true);
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  public Geometry fromWkt(String wkt) {
    try {
      return wktReader.read(wkt);
    } catch (Throwable t) {
      throw new RuntimeException(t);
    }
  }

  public Coordinate toCoordinate(double x, double y) {
    return new Coordinate(x, y);
  }

  public ReferencedEnvelope envelope(Coordinate lowerLeft, Coordinate upperRight) {
    return new ReferencedEnvelope(lowerLeft.x, upperRight.x, lowerLeft.y, upperRight.y, getCrs());
  }

  public Point emptyPoint() {
    return geometryFactory.createPoint((CoordinateSequence) null);
  }

  public Point point(double x, double y) {
    return geometryFactory.createPoint(toCoordinate(x, y));
  }

  public LineString emptyLine() {
    return geometryFactory.createLineString((CoordinateSequence) null);
  }

  public LineString line(Coordinate... coordinates) {
    return geometryFactory.createLineString(coordinates);
  }

  public LineString line(double x1, double y1, double x2, double y2) {
    return line(
        toCoordinate(x1, y1),
        toCoordinate(x2, y2)
    );
  }

  public Polygon emptyBox() {
    return geometryFactory.createPolygon((CoordinateSequence) null);
  }

  public Polygon box(Coordinate... coordinates) {
    return geometryFactory.createPolygon(coordinates);
  }

  public Polygon box(LinearRing shell, LinearRing... holes) {
    return geometryFactory.createPolygon(shell, holes);
  }

  public Polygon box(double x1, double y1, double x2, double y2) {
    return box(
        toCoordinate(x1, y1),
        toCoordinate(x2, y1),
        toCoordinate(x2, y2),
        toCoordinate(x1, y2),
        toCoordinate(x1, y1)
    );
  }

  public MultiPoint multiPoint(Point... points) {
    return geometryFactory.createMultiPoint(points);
  }

  public MultiLineString multiLine(LineString... lines) {
    return geometryFactory.createMultiLineString(lines);
  }

  public MultiPolygon multiBox(Polygon... boxes) {
    return geometryFactory.createMultiPolygon(boxes);
  }

  /**
   * Return a {@link GeometryCollection) containing geometries. Geometries can be a mixed bag of
   * different geometry types (lines, points and polygons).
   *
   * @param geometries
   * @return
   */
  public GeometryCollection collection(Geometry... geometries) {
    return new GeometryCollection(geometries, geometryFactory);
  }

  public LinearRing ring(Coordinate... coordinates) {
    return geometryFactory.createLinearRing(coordinates);
  }
}
