/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.gt;

import java.awt.Rectangle;
import java.awt.geom.Rectangle2D;
import java.awt.image.RenderedImage;

import org.geotools.coverage.grid.AbstractGridCoverage;
import org.geotools.coverage.grid.GridGeometry2D;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.geotools.api.coverage.CannotEvaluateException;
import org.geotools.api.coverage.PointOutsideCoverageException;
import org.geotools.api.coverage.SampleDimension;
import org.geotools.api.coverage.grid.GridGeometry;
import org.geotools.api.geometry.Position;
import org.geotools.api.geometry.Bounds;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

@SuppressWarnings("serial")
public class ArrayGridCoverage extends AbstractGridCoverage {

  private final double[][] grid;
  private final Rectangle size;

  public ArrayGridCoverage(final CharSequence name, final CoordinateReferenceSystem crs, Rectangle size) {
    super(name, crs, null, null);

    if (size.x != 0 || size.y != 0) {
      throw new IllegalArgumentException("only grids set at (0,0) are supported");
    }

    this.grid = new double[size.height][size.width];
    this.size = size;
  }

  protected Rectangle getRectangle() {
    return size;
  }

  @Override
  public GridGeometry getGridGeometry() {
    Rectangle gridRectangle = getRectangle();
    Rectangle2D  userRectangle = gridRectangle;
    return new GridGeometry2D(gridRectangle, userRectangle);
  }

  @Override
  public int[] getOptimalDataBlockSizes() {
    throw new RuntimeException("Not implemented");
  }

  @Override
  public RenderedImage getRenderedImage() {
    throw new RuntimeException("Not implemented");
  }

  @Override
  public Bounds getEnvelope() {
    return new ReferencedEnvelope(getRectangle(), crs);
  }

  public void fill(Rectangle rectangle, double value) {
    for (int x = rectangle.x; x < rectangle.x + rectangle.width; x++) {
      for (int y = rectangle.y; y < rectangle.y + rectangle.height; y++) {
        grid[y][x] = value;
      }
    }
  }

  @Override
  public Object evaluate(Position point) throws PointOutsideCoverageException, CannotEvaluateException {
    double[] coordinate = point.getCoordinate();

    // casting to int drops the decimal part of the double, which allows us to simply map the coordinates on to the
    // grid, ignoring any maths transform completely
    int x = (int) coordinate[0];
    int y = (int) coordinate[1];

    if (y >= grid.length) {
      throw new PointOutsideCoverageException();
    }

    double[] row = grid[y];

    if (x >= row.length) {
      throw new PointOutsideCoverageException();
    }

    return new double[] {grid[y][(int) x]};
  }

  @Override
  public int getNumSampleDimensions() {
    // TODO Auto-generated method stub
    return 0;
  }

  @Override
  public SampleDimension getSampleDimension(int index) throws IndexOutOfBoundsException {
    // TODO Auto-generated method stub
    return null;
  }
}
