/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli;

import java.io.File;
import java.util.AbstractMap;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Map.Entry;

import lombok.Getter;
import lombok.Setter;

public class RunParameters {
  // not a HashMap. we may want keys/first to be non-unique.
  private List<Entry<String,String>> params = new LinkedList<Entry<String,String>>();

  @Getter
  @Setter
  private BaseCliIntegrationTest test;

  public RunParameters(BaseCliIntegrationTest test) {
    this.test = test;
    clear();
  }

  public int size() {
    return params.size();
  }

  public Entry<String,String> get(int ctr) {
    return params.get(ctr);
  }

  /**
   * add a parameter whose value is not a path.
   * The parameter will become -${key}=${value}
   * e.g., ("out","data/output/theoutput.txt") will become
   * -out=data/output/theoutput.txt.
   *
   * This is used less often than add(String key) because this is also
   * perfectly fine and shorter:
   *
   * .add("-p=includeAssetInOutput=true")
   *
   * @param key
   * @param value
   * @return RunParameters, self.
   */
  private RunParameters add(String key, String value) {

    if(key.trim().endsWith("=")) {
      key=key.substring(0, key.length()-1);
    }

    Entry<String, String> ret = new AbstractMap.SimpleEntry<>(key, value);
    params.add(ret);

    return this;
  }

  /**
   * add a non-path parameter.  Often this is a bare argument that
   * takes no values
   * e.g., just "model" or just "function".
   *
   * But it's also fine to do this:
   * .add("-p=includeAssetInOutput=true")
   *
   * The output will just be model or just function, with no
   * equal sign or value on the RHS.
   *
   * @param keys
   * @return RunParameters
   */
  public RunParameters add(String... keys) {
    for (String key : keys) {
      add(key, null);
    }

    return this;
  }

  /*
   * Adds the given key and path to the parameters.  Since we don't
   * try to check existence of the path (nor do we generate canonical
   * paths, which would check existence) the pathP can include :, e.g.,
   * when setting up docker -v.
   *
   */
  public RunParameters addPath(String key, String...pathP) throws RuntimeException {
    List<String> path = Arrays.asList(pathP);

    String pathS = String.join(File.separator, path);

    // fix ...://home/riskscape
    pathS=pathS.replaceAll("//", "/");

    String value = pathS;

    // if windows, convert docker to riskscapeDir based.
    value = test.toRSDPath(value);

    add(key, value);
    return this;
  }

  public List<String> toList() {
    LinkedList<String> ret = new LinkedList<String>();

    for(Entry<String,String> pair : params) {
      String key = pair.getKey();
      String val = pair.getValue();

      String curr = key;
      if(null != val) {
        curr = key+"="+val;
      }

      ret.add(curr);
    }

    return ret;
  }

  // debug
  public String toString() {
    return toList().toString();
  }

  /**
   * The other parameters will be added at the end of THIS one.
   *
   * @param other
   * @return RunParameters
   */
  public RunParameters addAll(RunParameters other) {
    this.params.addAll(other.params);
    return this;
  }

  /*
   * Reset our internal parameters to empty.
   */
  public RunParameters clear() {
    params.clear();
    return this;
  }
}
