/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import nz.org.riskscape.engine.cli.BaseCliIntegrationTest.ExecResult;

/**
 * Convenience for forking processes and capturing their output
 */
public class ProcessRunner {

  public ExecResult exec(Path workDir,  List<String> commandList, Optional<Inputter> inputter) {
    return exec(workDir, commandList, inputter, Collections.emptyMap());
  }

  public ExecResult exec(Path workDir,  List<String> commandList, Optional<Inputter> inputter,
      Map<String, String> environmentVariables) {
    ProcessBuilder pbuilder = new ProcessBuilder();

    pbuilder.environment().putAll(environmentVariables);

    pbuilder.directory(workDir.toFile());

    ExecResult result = new ExecResult();


    System.out.println(commandList.stream().map(Object::toString).collect(Collectors.joining(" ")));

    pbuilder.command(commandList);
    try {
      Process process = pbuilder.start();

      inputter.ifPresent(i -> {
        i.start(process.getOutputStream());
      });

      List<Thread> threads = Arrays.asList(
        capture(result, "stderr", process.getErrorStream()),
        capture(result, "stdout", process.getInputStream()),
        inputter.orElse(null)
      );

      int retCode = process.waitFor();
      result.retCode = retCode;

      threads.forEach(t -> {
        try {
          if (t != null) {
            t.join();
          }
        } catch (InterruptedException e) {
          e.printStackTrace();
        }
      });
      return result;
    } catch (IOException e) {
      throw new RuntimeException(e);
    } catch (InterruptedException e) {
      e.printStackTrace();
      throw new RuntimeException(e);
    }
  }

  protected Thread capture(ExecResult result, String streamName, InputStream stream) {
    BufferedReader reader = new BufferedReader(new InputStreamReader(stream));
    Thread thread = new Thread(() ->  {
      try {
        String line = null;
        while ((line = reader.readLine()) != null) {
          System.err.println(String.format("[%s] %s", streamName, line));
          result.addOutput(streamName, line);
        }
      } catch (IOException e) {
        e.printStackTrace();
      }

    });

    thread.start();

    return thread;
  }

}
