/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import org.hamcrest.Description;
import org.hamcrest.DiagnosingMatcher;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.engine.types.Type;

public class TupleMatchers implements MatcherCollection {

  /**
   * Type-checking matcher when the thing you're matching against might be in a collection of anonymous objects and you
   * want to assert an element is a {@link Tuple}  before handing over to a more useful matcher.
   */
  public static Matcher<? extends Object> isTuple(Matcher<Tuple> tupleMatcher) {
    return new DiagnosingMatcher<Object>() {

      @Override
      protected boolean matches(Object item, Description mismatchDescription) {
        if (item instanceof Tuple) {
          if (tupleMatcher.matches(item)) {
            return true;
          } else {
            tupleMatcher.describeMismatch(item, mismatchDescription);
            return false;
          }
        } else {
          mismatchDescription.appendText("item not a tuple - was ")
              .appendValue(item == null ? "null" : item.getClass());
          return false;
        }
      }

      @Override
      public void describeTo(Description description) {
        tupleMatcher.describeTo(description);
      }
    };
  }

  /**
   * A matcher that will match a tuple that has a member with a value that matches the given matcher.
   */
  public static Matcher<Tuple> tupleWithValue(String member, Matcher<?> valueMatcher) {
    return new TypeSafeDiagnosingMatcher<Tuple>() {

      @Override
      public void describeTo(Description description) {
        description.appendText("tuple with member ").appendText(member).appendText(" and value ")
          .appendDescriptionOf(valueMatcher);
      }

      @Override
      protected boolean matchesSafely(Tuple item, Description mismatchDescription) {
        if (item.getStruct().hasMember(member)) {
          Object value = item.fetch(member);
          if (valueMatcher.matches(value)) {
            return true;
          } else {
            mismatchDescription.appendText("value of ").appendValue(member).appendText(" does not match");
            valueMatcher.describeMismatch(value, mismatchDescription);
            return false;
          }
        } else {
          mismatchDescription.appendText("tuple has no member ").appendText(member);
          return false;
        }
      }
    };
  }

  /**
   * A matcher that will match a tuple that has a member that matches the given type matcher.
   */
  public static Matcher<Tuple> tupleWithMemberType(String memberName, Matcher<Type> typeMatcher) {
    return new TypeSafeDiagnosingMatcher<Tuple>() {

      @Override
      public void describeTo(Description description) {
        description.appendText("tuple with member ").appendText(memberName).appendText(" and type ")
          .appendDescriptionOf(typeMatcher);
      }

      @Override
      protected boolean matchesSafely(Tuple item, Description mismatchDescription) {
        if (item.getStruct().getMember(memberName).isPresent()) {
          StructMember member = item.getStruct().getEntry(memberName);
          if (typeMatcher.matches(member.getType())) {
            return true;
          } else {
            mismatchDescription.appendText("member of ").appendValue(memberName).appendText(" does not match");
            typeMatcher.describeMismatch(member.getType(), mismatchDescription);
            return false;
          }
        } else {
          mismatchDescription.appendText("tuple has no member ").appendText(memberName);
          return false;
        }
      }
    };
  }
}
