/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.stream.Collectors;

/**
 * Contains helpers for executing tasks concurrently.
 */
public interface TaskHelper {

  /**
   * Convenience version of {@link #executeTasks(java.util.List, int) } where the thread pool size
   * equals the number of tasks to execute.
   */
  default <T> List<T> executeTasks(List<Callable<T>> tasks) {
    return executeTasks(tasks, tasks.size());
  }

  /**
   * Executes the given tasks returning once all tasks are complete.
   *
   * @param <T>
   * @param tasks the tasks to execute
   * @param threadPoolSize the size of the thread pool to use when executing tasks
   * @return list containing a result for each task
   */
  default <T> List<T> executeTasks(List<Callable<T>> tasks, int threadPoolSize) {
    ExecutorService executor = Executors.newFixedThreadPool(threadPoolSize);
    try {
      List<Future<T>> futures = executor.invokeAll(tasks);
      while (futures.stream().anyMatch(f -> !f.isDone())) {
        Thread.sleep(100);
      }
      return futures.stream()
          .map(task -> {
            try {
              return task.get();
            } catch (Exception e) {
              throw new RuntimeException(e);
            }
          })
          .collect(Collectors.toList());
    } catch (InterruptedException e) {
      throw new RuntimeException(e);
    }
  }

}
