/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import org.geotools.api.referencing.crs.CoordinateReferenceSystem;
import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;
import org.hamcrest.TypeSafeMatcher;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.PrecisionModel;

public class GeometryMatchers implements MatcherCollection {

  /**
   * Match a specific coordinate within a tolerance.  Tolerance works the same way as with the `assertEquals(double)`
   * methods
   */
  public static Matcher<Coordinate> coordEqual(Coordinate expected, double tolerance) {
    return new TypeSafeDiagnosingMatcher<Coordinate>(Coordinate.class) {
      @Override
      public void describeTo(Description description) {
        description.appendText("coord ").appendValue(expected).appendText(" within ").appendValue(tolerance);
      }

      @Override
      protected boolean matchesSafely(Coordinate item, Description mismatchDescription) {

        double xDelta = Math.abs(item.getX() - expected.getX()) - tolerance;
        double yDelta = Math.abs(item.getY() - expected.getY()) - tolerance;

        if (xDelta <= 0 && yDelta <= 0) {
          return true;
        } else {
          mismatchDescription.appendValue(item).appendText(" not close enough to ").appendValue(expected);
          return false;
        }
      }
    };
  }

  /**
   * A {@link Geometry} matcher.
   *
   * This matcher will test geometry equality using
   * {@link Geometry#equalsNorm(org.locationtech.jts.geom.Geometry) so that two polygons that have the
   * same coordinates but with a different start point or direction to the coordinate order are matched.
   *
   */
  public static Matcher<Geometry> isGeometry(Geometry toMatch) {
    return new TypeSafeMatcher<Geometry>() {
      @Override
      protected boolean matchesSafely(Geometry t) {
        return toMatch.equalsNorm(t);
      }

      @Override
      public void describeTo(Description d) {
        d.appendText("Geometry: " + toMatch.toString());
      }

    };
  }

  /**
   * A Matcher that will match a geometry that has an srid value which equates to the given crs according to the given
   * srid set.
   */
  public static Matcher<Geometry> isInCrs(SRIDSet sridSet, CoordinateReferenceSystem crs) {
    int expectedSrid = sridSet.get(crs);
    return new TypeSafeDiagnosingMatcher<Geometry>(Geometry.class) {

      @Override
      public void describeTo(Description description) {
        description.appendText("geometry in crs ").appendValue(crs);
      }

      @Override
      protected boolean matchesSafely(Geometry item, Description mismatchDescription) {
        int actualSrid = item.getSRID();
        if (actualSrid == expectedSrid) {
          return true;
        } else {
          CoordinateReferenceSystem actualCrs = sridSet.get(actualSrid);
          mismatchDescription.appendText("crs was ").appendValue(actualCrs);
          return false;
        }
      }
    };
  }

  public static Matcher<Geometry> geometryWithSrid(int expectedSrid) {

    return new TypeSafeDiagnosingMatcher<Geometry>(Geometry.class) {

      @Override
      public void describeTo(Description description) {
        description.appendText("geometry with srid").appendValue(expectedSrid);
      }

      @Override
      protected boolean matchesSafely(Geometry item, Description mismatchDescription) {
        int actualSrid = item.getSRID();
        if (actualSrid == expectedSrid) {
          return true;
        } else {
          mismatchDescription.appendText("srid was ").appendValue(actualSrid);
          return false;
        }
      }
    };
  }

  public static Matcher<Geometry> geometryTopoMatch(Geometry expected) {
    return GeoHelper.geometryTopoMatch(expected);
  }

  public static Matcher<Geometry> geometryTopoMatch(Geometry expected, int precisionScale) {
    return GeoHelper.geometryTopoMatch(expected, new PrecisionModel(precisionScale));
  }
}
