/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import java.io.File;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

public class FileSystemMatchers implements MatcherCollection {

  public static Matcher<Path> hasFile(Matcher<File> fileMatcher) {
    return new TypeSafeDiagnosingMatcher<Path>(Path.class) {

      @Override
      public void describeTo(Description description) {
        description.appendText("directory with file matching");
        fileMatcher.describeTo(description);
      }

      @Override
      protected boolean matchesSafely(Path item, Description mismatchDescription) {
        File file = item.toFile();

        if (!file.exists()) {
          mismatchDescription.appendValue(item).appendText(" does not exist");
          return false;
        }

        if (!file.isDirectory()) {
          mismatchDescription.appendValue(item).appendText(" not a directory");
          return false;
        }

        List<File> files = Arrays.asList(file.listFiles());
        for (File fileInDir : files) {
          if (fileMatcher.matches(fileInDir)) {
            return true;
          }
        }

        mismatchDescription.appendValue(item).appendText(" did not have a file matching ");
        mismatchDescription.appendDescriptionOf(fileMatcher);
        mismatchDescription.appendText(" among ");
        mismatchDescription.appendValue(
        files.stream().map(f -> f.getName()).limit(10).collect(Collectors.toList()));

        return false;
      }
    };
  }

  public static Matcher<File> fileWithName(String string) {
    return new TypeSafeDiagnosingMatcher<File>(File.class) {

      @Override
      public void describeTo(Description description) {
        description.appendText(" accessible file named " + string);
      }

      @Override
      protected boolean matchesSafely(File item, Description mismatchDescription) {
        if (item.getName().equals(string)) {
          if (item.canRead()) {
            return true;
          } else {
            mismatchDescription.appendValue(item.getName()).appendText(" was not readable");
            return false;
          }
        } else {
          mismatchDescription.appendText("was named " + item.getName());
          return false;
        }
      }
    };
  }

}
