/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;

import org.geotools.referencing.CRS;
import org.geotools.api.referencing.FactoryException;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

public interface CrsHelper {

  /**
   * @return the {@link CoordinateReferenceSystem} for EPSG:27200 parsed from WKT.
   */
  default CoordinateReferenceSystem nzMapGrid() {
    return crsFromClassPathResource("/nz/org/riskscape/engine/crs/EPSG27200.wkt");
  }

  /**
   * @return the {@link CoordinateReferenceSystem} for EPSG:2193 parsed from WKT
   */
  default CoordinateReferenceSystem nzTransverseMercator() {
    return crsFromClassPathResource("/nz/org/riskscape/engine/crs/EPSG2193.wkt");
  }

  /**
   * @return the {@link CoordinateReferenceSystem} for EPSG:4326 with long/lat (XY) axis order, parsed from WKT
   */
  default CoordinateReferenceSystem longLat() {
    return crsFromClassPathResource("/nz/org/riskscape/engine/crs/EPSG4326.wkt");

  }

  /**
   * @return A custom CRS, it's similar to EPSG:2193 (with YX axis) but shunted over a bit. Exists
   * to test shapefile writer code with a YX projection that isn't from an EPSG code.
   */
  default CoordinateReferenceSystem customCrs() {
    return crsFromClassPathResource("/nz/org/riskscape/engine/crs/custom.wkt");
  }

  /**
   * Get a {@link CoordinateReferenceSystem} that is parsed from filename, which is expected to contain
   * WKT and be found on the classpath in `/nz/org/riskscape/engine/crs/`.
   * @param filename  name for file containing WKT to parse
   * @return CRS from WKT
   */
  default CoordinateReferenceSystem crsFromWkt(String filename) {
    return crsFromClassPathResource("/nz/org/riskscape/engine/crs/" + filename);
  }

  default CoordinateReferenceSystem crsFromClassPathResource(String classpath) {
    InputStream resourceAsStream = getClass().getResourceAsStream(classpath);
    StringBuilder builder = new StringBuilder();
    BufferedReader reader = new BufferedReader(new InputStreamReader(resourceAsStream));
    String line;
    try {
      while ((line = reader.readLine()) != null) {
        builder.append(line);
      }
    } catch (IOException e) {
      throw new RuntimeException(e);
    }

    try {
      return CRS.parseWKT(builder.toString());
    } catch (FactoryException e) {
      throw new RuntimeException(e);
    }
  }
}
