/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.test.functions;

import static nz.org.riskscape.test.functions.SimpleMathFunction.Operation.*;

import java.util.List;
import java.util.function.BiFunction;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.bind.ParameterField;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.FunctionArgument;
import nz.org.riskscape.engine.function.FunctionContext;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.RealizableFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.util.FunctionCallOptions;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

import lombok.Getter;

/**
 * Simple function that does some basic math.
 *
 * Intended as a worked example of how to write a realizable RiskScape function in java.
 */
public class SimpleMathFunction implements IdentifiedFunction, RealizableFunction {

  public enum Operation {
    ADD,
    SUBTRACT,
    MUTLIPLY,
    DIVIDE
  }

  public static class Options {

    @ParameterField
    public Operation operation = ADD;
  }

  @Getter
  public final String id = "simple_math";

  @Getter
  public final String description = "function for doing some simple math operations";

  @Getter
  public final String source = "test-functions";

  @Getter
  public final Category category = Category.UNASSIGNED;

  @Getter
  public final ArgumentList arguments = ArgumentList.fromArray(
      new FunctionArgument("a", Types.FLOATING),
      new FunctionArgument("b", Types.FLOATING),
      FunctionCallOptions.options(Options.class)
  );

  @Getter
  public final Type returnType = Types.FLOATING;

  public SimpleMathFunction(FunctionContext functionContext) {}


  @Override
  public List<Type> getArgumentTypes() {
    return arguments.getArgumentTypes();
  }

  @Override
  public Object call(List<Object> args) {
    // this method is never called on a realizable function
    throw new UnsupportedOperationException("Not supported yet.");
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenTypes) {
    return ProblemException.catching(() -> {
      if (givenTypes.size() > arguments.size() || givenTypes.size() < arguments.size() - 1) {
        throw new ProblemException(
            ArgsProblems.get().wrongNumberRange(arguments.size() - 1, arguments.size(), givenTypes.size())
        );
      }
      if (! givenTypes.get(0).isNumeric()) {
        throw new ProblemException(ArgsProblems.mismatch(arguments.get(0), givenTypes.get(0)));
      }
      if (! givenTypes.get(1).isNumeric()) {
        throw new ProblemException(ArgsProblems.mismatch(arguments.get(1), givenTypes.get(1)));
      }

      Options options = FunctionCallOptions.bindOptionsOrThrow(Options.class, context, arguments, functionCall);
      BiFunction<Double, Double, Double> func;
      if (options.operation == ADD) {
        func = (a, b) -> a + b;
      } else if (options.operation == SUBTRACT) {
        func = (a, b) -> a - b;
      } else if (options.operation == MUTLIPLY) {
        func = (a, b) -> a * b;
      } else  {
        func = (a, b) -> a / b;
      }

      return new RiskscapeFunction() {
        @Override
        public List<Type> getArgumentTypes() {
          return givenTypes;
        }

        @Override
        public Type getReturnType() {
          return Types.FLOATING;
        }

        @Override
        public Object call(List<Object> args) {
          Number a = (Number)args.get(0);
          Number b = (Number)args.get(1);

          return func.apply(a.doubleValue(), b.doubleValue());
        }

      };
    });
  }

}
