/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.plugin;


import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Collections;

import org.junit.Test;

import nz.org.riskscape.engine.cli.BaseCliIntegrationTest;
import nz.org.riskscape.engine.cli.RunParameters;

public class PluginTest extends BaseCliIntegrationTest {

  private RunParameters listPluginsCmd() {
    // the --version command lists all the loaded plugins
    return params.clear().add("--version");
  }

  @Test
  public void byDefaultRiskscapeLoadsCorePlugins() {
    this.plugins = Collections.emptyList();
    this.riskscapeDir = expand("src/test/homes/plugins/empty");
    ExecResult result = execute(listPluginsCmd());

    assertThat(result.stdout, hasItems(
        containsString("nz.org.riskscape.engine.defaults.Plugin"),
        containsString("nz.org.riskscape.jython.Plugin"),
        containsString("nz.org.riskscape.postgis.Plugin")
    ));
  }

  @Test
  public void canTellRiskscapeNotToLoadCorePlugins() {
    this.plugins = Collections.emptyList();
    this.riskscapeDir = expand("src/test/homes/plugins/empty");
    ExecResult result = execute(listPluginsCmd().add("--no-core-plugins"));

    assertThat(result.stdout, not(hasItems(
        containsString("nz.org.riskscape.engine.defaults.Plugin"),
        containsString("nz.org.riskscape.jython.Plugin"),
        containsString("nz.org.riskscape.postgis.Plugin")
    )));
  }

  @Test
  public void canTellRiskscapeWhatPluginsToLoadInSettingsFile() {
    this.plugins = Collections.emptyList();
    this.riskscapeDir = expand("src/test/homes/plugins/from-settings-file");
    ExecResult result = execute(listPluginsCmd());

    assertThat(result.stdout, hasItems(
        containsString("nz.org.riskscape.engine.defaults.Plugin"),
        containsString("nz.org.riskscape.cpython.CPythonPlugin")
    ));

    assertThat(result.stdout, not(hasItems(
        containsString("nz.org.riskscape.jython.Plugin"),
        containsString("nz.org.riskscape.postgis.Plugin")
    )));
  }

  @Test
  public void canTellRiskscapeWhatPluginsToLoadInSettingsFileWithPluginNames() {
    this.plugins = Collections.emptyList();
    // the settings file in this home directory has plugins names (not paths)
    this.riskscapeDir = expand("src/test/homes/plugins/settings-file-by-name");
    ExecResult result = execute(listPluginsCmd());

    assertThat(result.stdout, hasItems(
        containsString("nz.org.riskscape.engine.defaults.Plugin"),
        containsString("nz.org.riskscape.cpython.CPythonPlugin"),
        containsString("nz.org.riskscape.netcdf.Plugin")
    ));

    assertThat(result.stdout, not(hasItems(
        containsString("nz.org.riskscape.jython.Plugin"),
        containsString("nz.org.riskscape.postgis.Plugin")
    )));
  }

  @Test
  public void RM212_willNotPrintWarningsIfHomePluginsDirectoryIsMissing() {
    this.riskscapeDir = expand("src/test/homes/plugins/empty-sans-plugins");
    this.plugins = Collections.emptyList();
    ExecResult result = execute(listPluginsCmd().add("--no-core-plugins"));

    assertEquals(0, result.retCode.intValue());
    assertTrue(result.stderr.isEmpty());
  }

  @Test
  public void canTellRiskscapeToLoadSpecificPlugins() {
    this.riskscapeDir = expand("src/test/homes/plugins/empty");
    this.plugins = Collections.emptyList();
    ExecResult result = execute(listPluginsCmd()
        .add("--no-core-plugins")
        .addPath("-l", "/opt", "riskscape-cli", "plugins", "defaults"));

    assertEquals(0, result.retCode.intValue());

    assertThat(result.stdout, hasItems(
        containsString("nz.org.riskscape.engine.defaults")
    ));

    assertThat(result.stdout, not(hasItems(
        containsString("nz.org.riskscape.jython.Plugin")
    )));
  }

  @Test
  public void pluginsFromHomeDirectoryAreIgnored() {
    // this has a plugins dir, which is "good enough" as it follows the same shape as the home/plugins folder
    this.riskscapeDir = expand("../../engine/app/build/install/riskscape");
    this.plugins = Collections.emptyList();
    ExecResult result = execute(listPluginsCmd().add("--no-core-plugins"));

    assertEquals(0, result.retCode.intValue());

    // we used to warn, but that's gone away - users have had about a year now :)
    assertThat(result.stderr, not(contains(
        containsString("[WARNING] Home directory plugins are no longer loaded by RiskScape")
    )));

    assertThat(result.stdout, not(hasItems(
        containsString("nz.org.riskscape.engine.defaults.Plugin"),
        containsString("nz.org.riskscape.jython.Plugin"),
        containsString("nz.org.riskscape.postgis.Plugin")
    )));
  }

}
