/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.cli.tests;

import static nz.org.riskscape.engine.Assert.*;

import static org.junit.Assert.*;

import java.nio.file.Files;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.util.FileUtils;
import nz.org.riskscape.problem.Problem;

/**
 * Confirms that temporary files are deleted by the execution context cleanup.
 */
public class TempFilesAreCleanedAwayTest extends BaseModelRunCommandTest{

  @Before
  public void emptyTempDir() throws Exception {
    // clean away any previous contents of the temp directory. Other tests may have saved
    // stuff in here and we don't want it clouding the waters.
    // Note that these tests share an engine with other tests with the same set of plugins.
    FileUtils.cleanDirectory(engine.getTempDirectory());
  }

  @Test
  public void successfulModelRunLeavesNoTempFiles() throws Exception {
    runCommand.modelId = "remote";
    runCommand.run();

    // we don't really care too much about the results. we only care about temp dir usage
    assertEquals(0L, Files.list(engine.getTempDirectory()).count());

  }

  @Test
  public void failledModelRunLeavesNoTempFiles() throws Exception {
    runCommand.modelId = "remote-fails";
    RiskscapeException expected = assertThrows(RiskscapeException.class, () -> runCommand.run());

    // we don't really care too much about the results. we only care about temp dir usage
    assertEquals(0L, Files.list(engine.getTempDirectory()).count());
  }

  @Test
  public void pipelineLeavesNoTempFiles() throws Exception {
    evalCommand.pipelineFile = "input('remote') -> save('output', 'csv')";
    evalCommand.run();

    // we don't really care too much about the results. we only care about temp dir usage
    assertEquals(0L, Files.list(engine.getTempDirectory()).count());
  }

  @Test
  public void failledPipelineLeavesNoTempFiles() throws Exception {
    evalCommand.pipelineFile = "input('remote') -> select({unknown as missing}) -> save('output', 'csv')";
    Object result =  evalCommand.run();
    assertTrue(result instanceof Problem);

    // we don't really care too much about the results. we only care about temp dir usage
    assertEquals(0L, Files.list(engine.getTempDirectory()).count());
  }
}
