/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;


import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;

import org.junit.Test;

import nz.org.riskscape.engine.test.EngineTestPlugins;

/**
 * Exercise the subpipelines examples in the docs (docs/source/reference/subpipelines.md) to check they
 * work as documented.
 */
@EngineTestPlugins({ "defaults", "beta" })
public class SubpipelinesTutorialTest extends BaseModelRunCommandTest {

  @Override
  public Path stdhome() {
    return Paths.get("..", "..", "examples", "subpipelines");
  }

  @Test
  public void canEvalHelloWorld() throws Exception {
    evalCommand.pipelineFile = "subpipeline('hello_world.txt')";
    evalCommand.doCommand(project);

    List<List<String>> results = openCsv("subpipeline.csv", "greeting");

    assertThat(results, contains(
        Arrays.asList("Hello, world x1"),
        Arrays.asList("Hello, world x2"),
        Arrays.asList("Hello, world x3")
    ));
  }

  @Test
  public void canEvalHelloWorldParams() throws Exception {
    evalCommand.pipelineFile = "subpipeline('hello_world_params.txt', parameters: { who: 'Bob', number: 2})";
    evalCommand.doCommand(project);

    List<List<String>> results = openCsv("subpipeline.csv", "greeting");

    assertThat(results, contains(
        Arrays.asList("Hello, Bob x1"),
        Arrays.asList("Hello, Bob x2")
    ));
  }

  @Test
  public void canEvalHelloWorldInput() throws Exception {
    evalCommand.pipelineFile = """
                               input(value: { who: 'Bob', number: 2})
                               -> subpipeline('hello_world_input.txt')
                               """;
    evalCommand.doCommand(project);

    List<List<String>> results = openCsv("subpipeline.csv", "greeting");

    assertThat(results, contains(
        Arrays.asList("Hello, Bob x1"),
        Arrays.asList("Hello, Bob x2")
    ));
  }

  @Test
  public void canRunExposedPopulationModel() throws Exception {
    runCommand.modelId = "Exposed-Population";

    runCommand.doCommand(project);

    List<List<String>> results = openCsv("regional-summary.csv", "fid", "NAME_1", "Region", "Population",
        "Number_Households", "Residential_floor_area", "Exposed_buildings", "Exposed_population",
        "Depth.range_<_1.aggregated.Exposed_buildings", "Depth.range_<_1.aggregated.Exposed_population",
        "Depth.range_1_2.aggregated.Exposed_buildings", "Depth.range_1_2.aggregated.Exposed_population",
        "Depth.range_2_3.aggregated.Exposed_buildings", "Depth.range_2_3.aggregated.Exposed_population",
        "Depth.range_3_+.aggregated.Exposed_buildings", "Depth.range_3_+.aggregated.Exposed_population");

    // spot check one of the result lines
    assertThat(results, hasItem(
        Arrays.asList("7", "Atua", "Aleipata Itupa i Lalo", "4532.0", "640", "79051", "382", "2063", "259",
            "1449", "72", "320", "50", "294", "1", "0")
    ));
  }

}
