/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import java.nio.file.Paths;
import java.util.List;

import nz.org.riskscape.engine.cli.BaseCliIntegrationTest.ExecResult;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.hasItem;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

import java.util.Arrays;
import java.util.Map;

import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;

import com.google.common.collect.ImmutableMap;

@Ignore
public class RM313_NetCDF_Support extends ModelPipelineIntegrationCliTest {

  Struct expectedType;
  Struct outputType;

  @Before
  public void setupTypes() {
    outputType = Struct.of("asset-desc", Types.TEXT, "hazard-value", Types.INTEGER);
    expectedType = Struct.of("output", outputType);
  }

  @Test
  public void canRunAgainstSingleLayerNetCDFHazard() throws Exception {
    ExecResult result = execute(params.clear()
        .add("--project=project.ini")
        .add("--load-plugin", getInstallDir().resolve(Paths.get("plugins-optional", "netcdf")).toString())
        .add("model")
        .add("run")
        .add("loss")
        .add("-p=hazard='layer1'")
        .add("--output=output")
        .add("--replace"));

    assertEquals(0, result.retCode.intValue());

    Map<String, List<String>> expected = ImmutableMap.of(
        "2", Arrays.asList("black1", "1"),
        "3", Arrays.asList("black2", "1"),
        "4", Arrays.asList("grey1", "2")
    );
    assertHasRecords(expected, outputDir.resolve("results.csv"), "exposure.id",
        "exposure.desc", "consequence.hazard-value");
  }

  @Test
  public void canRunAgainstMultiLayerNetCDFHazard() throws Exception {
    doRunAgainstMultiLayerNetCDFHazard("band1", 1);
    doRunAgainstMultiLayerNetCDFHazard("band2", 10);
    doRunAgainstMultiLayerNetCDFHazard("band3", 100);
  }

  public void doRunAgainstMultiLayerNetCDFHazard(String coverageName, int multiplier) throws Exception {
    ExecResult result = execute(params.clear()
        .add("--project=project.ini")
        .add("--load-plugin", getInstallDir().resolve(Paths.get("plugins-optional", "netcdf")).toString())
        .add("model")
        .add("run")
        .add("loss")
        .add("-p=hazard=" + String.format("'layer-%s'", coverageName))
        .add("--output=output")
        .add("--replace"));

    assertEquals(0, result.retCode.intValue());

    Map<String, List<String>> expected = ImmutableMap.of(
        "2", Arrays.asList("black1", String.valueOf(1 * multiplier)),
        "3", Arrays.asList("black2", String.valueOf(1 * multiplier)),
        "4", Arrays.asList("grey1", String.valueOf(2 * multiplier))
    );
    assertHasRecords(expected, outputDir.resolve("results.csv"), "exposure.id",
        "exposure.desc", "consequence.hazard-value");
  }

  @Test
  public void canRunAgainstMultiLayerNetCDFHazardNoCoverageNameUsesFirstCoverage() throws Exception {
    ExecResult result = execute(params.clear()
        .add("--project=project.ini")
        .add("--load-plugin", getInstallDir().resolve(Paths.get("plugins-optional", "netcdf")).toString())
        .add("model")
        .add("run")
        .add("loss")
        .add("-p=hazard='layers'")
        .add("--output=output")
        .add("--replace"));

    assertEquals(0, result.retCode.intValue());

    Map<String, List<String>> expected = ImmutableMap.of(
        "2", Arrays.asList("black1", "1"),
        "3", Arrays.asList("black2", "1"),
        "4", Arrays.asList("grey1", "2")
    );
    assertHasRecords(expected, outputDir.resolve("results.csv"), "exposure.id",
        "exposure.desc", "consequence.hazard-value");
  }

  @Test
  public void youGetAnErrorIfRequestedCovererageDoesNotExist() {
    ExecResult result = execute(params.clear()
        .add("--project=project.ini")
        .add("--load-plugin", getInstallDir().resolve(Paths.get("plugins-optional", "netcdf")).toString())
        .add("model")
        .add("run")
        .add("loss")
        .add("-p=hazard='layer-bogus'")
        .add("--output=output"));

    assertEquals(1, result.retCode.intValue());
    assertThat(result.stderr, hasItem(
        containsString("Layer 'bogus' is not a known layer. Available layers are [Band1, Band2, Band3]")));
  }

}
