/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Test;

import nz.org.riskscape.engine.data.InputDataProblems;

public class RM312_RelationProblemsAreLogged extends BaseModelRunCommandTest {

  @Override
  public Path stdhome() {
    return homes().resolve("rm312-relation-problems-are-logged/");
  }

  @Test
  public void badCsvRowsAreLogged() throws Exception {
    runCommand.modelId = "input-only";
    runCommand.run();

    List<List<String>> results = openCsv("output.csv", "foo", "bar", "baz", "geom");
    assertThat(results, containsInAnyOrder(
        Arrays.asList("4", "5", "6", "POINT (4 5)"),
        Arrays.asList("6", "5", "4", "POINT (6 5)")
    ));

    List<String> problems = collectedSinkProblems.stream()
        .map(problem -> render(problem))
        .collect(Collectors.toList());

    assertThat(problems, containsInAnyOrder(
        containsString("malformed.csv:3 contained wrong number of columns - was 4, expected 3"),
        containsString("malformed.csv:5 contained wrong number of columns - was 2, expected 3")
    ));
  }

  @Test
  public void typeCoercionProblemsAreLogged() throws Exception {
    runCommand.modelId = "input-only";
    runCommand.parameters = Arrays.asList("my_input='buildings'");
    runCommand.run();

    List<List<String>> results = openCsv("output.csv", "geom", "street", "floors");
    assertThat(results, containsInAnyOrder(
        Arrays.asList("POINT (1567346.232323 8765412.4442)", "4 messington street", "4")
    ));

    String invalidTupleSkipped = render(InputDataProblems.get().invalidTupleSkipped());

    List<String> problems = collectedSinkProblems.stream()
        .map(problem -> render(problem))
        .collect(Collectors.toList());

    assertThat(problems, containsInAnyOrder(
        allOf(
            containsString(invalidTupleSkipped),
            containsString("Failed to evaluate expression 'create_point(x, y)' against {x=, y=, "
                + "street_name=66 sesame street, num_floors=1}")
        ),
        allOf(
            containsString(invalidTupleSkipped),
            containsString("Failed to coerce 'eleven' for floors[Integer] from {x=4, y=5, "
                + "street_name=9 coronation street, num_floors=eleven, geom=POINT (4 5)}")

        ),
        allOf(
            containsString(invalidTupleSkipped),
            containsString("Failed to evaluate expression 'create_point(x, y)' against {x=66, y=, "
                + "street_name=10 downing street, num_floors=3} for attribute geom[Point]")

        )
    ));
  }

}
