/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.Set;

import org.junit.Test;

import nz.org.riskscape.engine.FileSystemMatchers;

/**
 * Runs some of the documented pipeline examples to verify that they continue to work
 */
public class PipelineExamplesTest extends BaseModelRunCommandTest {

  @Override
  public Path stdhome() {
    return homes().resolve("pipeline-advanced");
  }

  @Override
  public Path homes() {
    return Paths.get("..", "..", "examples");
  }

  @Test
  public void canRunTheAdvancedPipelineTutorial() throws Exception {
    // answers.txt is the final version of the pipeline (including 'Additional exercises' section)
    evalCommand.pipelineFile = stdhome().resolve("answers.txt").toString();

    evalCommand.run();

    assertThat(getTempDirectory(), FileSystemMatchers.hasFile(
        FileSystemMatchers.fileWithName("regional-exposure.csv")
    ));

    // spot-check that the correct data is coming out
    Set<List<String>> rows = openCsvUniqueData("regional-exposure.csv", "consequence", "Cons_Frame",
        "Number of buildings", "Median_depth_cm");
    assertThat(rows, hasItem(
        hasItems("Exposed", "Masonry", "1549", "220")
    ));
  }

  @Test
  public void canRunThePipelineParametersExample() throws Exception {
    populateProjectAndSetupCommands(homes().resolve("pipeline-parameters"));
    runCommand.modelId = "demo";

    runCommand.run();

    assertThat(getTempDirectory(),
        FileSystemMatchers.hasFile(FileSystemMatchers.fileWithName("results.csv")));
    assertThat(getTempDirectory(),
        FileSystemMatchers.hasFile(FileSystemMatchers.fileWithName("by_construction.csv")));
    assertThat(getTempDirectory(),
        FileSystemMatchers.hasFile(FileSystemMatchers.fileWithName("pipeline.txt")));

    // spot-check that the correct data is coming out
    Set<List<String>> rows = openCsvUniqueData("by_construction.csv", "event_id", "damage.stone.total",
        "damage.steel.total", "damage.concrete.total");
    assertThat(rows, hasItem(hasItems("3", "100.0", "171.5", "99.0")));
  }

  @Test
  public void canRunThePipelineParametersExampleWithIniParametersFile() throws Exception {
    Path projectPath = homes().resolve("pipeline-parameters");
    populateProjectAndSetupCommands(projectPath);
    runCommand.modelId = "demo";
    runCommand.parametersFile = projectPath.resolve("stronger-resilience.ini");
    runCommand.run();

    assertThat(getTempDirectory(), FileSystemMatchers.hasFile(FileSystemMatchers.fileWithName("results.csv")));
    assertThat(getTempDirectory(), FileSystemMatchers.hasFile(FileSystemMatchers.fileWithName("by_construction.csv")));
    assertThat(getTempDirectory(), FileSystemMatchers.hasFile(FileSystemMatchers.fileWithName("pipeline.txt")));

    // spot-check that the correct data is coming out
    Set<List<String>> rows = openCsvUniqueData("by_construction.csv", "event_id", "damage.stone.total",
        "damage.steel.total", "damage.concrete.total");
    // the INI params assume buildings are more resilient, so damage is less this time
    assertThat(rows, hasItem(hasItems("3", "92.0", "73.5", "55.0")));
  }
}
