/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static nz.org.riskscape.engine.Assert.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.nio.file.Files;
import java.nio.file.Path;

import org.junit.Test;

import com.google.common.collect.Lists;

import nz.org.riskscape.engine.cli.ExitException;
import nz.org.riskscape.problem.Problem;

public class PipelineCommandEvalTest extends BaseModelRunCommandTest {

  @Test
  public void canParsePipelineFromFile() throws Exception {
    evalCommand.pipelineFile = stdhome().resolve("pipeline.txt").toString();
    evalCommand.run();
    assertThat(readCsvColumns("input.csv", "foo"), contains(Lists.newArrayList("bar")));

    // check that the pipeline file has some content. we're just checking it isn't empty here
    assertThat(openFile("pipeline.txt"), containsString("input"));
  }

  @Test
  public void givesStdinErrorIfFailedToParseStdinPipeline() throws ExitException {
    setCommandInput("input(value: 'foo'");
    evalCommand.pipelineFile = "-";
    Problem problem = assertThrows(ExitException.class, () -> evalCommand.run()).getProblem();

    assertThat(render(problem), allOf(
        containsString("Failed to parse pipeline from STDIN"),
        containsString("Got '', wanted one of [")
    ));
  }

  @Test
  public void canCreatePipelineFromSTDIN() throws Exception {
    setCommandInput("input(value: {foo: 'bar'})");
    evalCommand.pipelineFile = "-";
    evalCommand.run();
    assertThat(readCsvColumns("input.csv", "foo"), contains(Lists.newArrayList("bar")));
  }

  @Test
  public void canSavePipelineAstRun() throws Exception {
    // note that AST is not saved with STDIN, but that's too much of a corner-case to bother with
    evalCommand.pipelineFile = "input(value: {foo: 'bar'})";
    evalCommand.run();

    Path savedPipelineAst = getOutputPath("pipeline.txt");

    assertThat(
        new String(Files.readAllBytes(savedPipelineAst)).trim(),
        is("input(value: {foo: 'bar'})")
    );
  }

  @Test
  public void givesFileErrorMessageIfStartsWithSlashes() {
    evalCommand.pipelineFile = "./foo.txt";
    Problem problem = assertThrows(ExitException.class, () -> evalCommand.run()).getProblem();
    assertThat(render(problem), allOf(
        containsString("Failed to load pipeline from file"),
        containsString(" - File not found at /"),
        containsString("./foo.txt")
    ));

    evalCommand.pipelineFile = "/foo.txt";
    problem = assertThrows(ExitException.class, () -> evalCommand.run()).getProblem();
    assertThat(render(problem), allOf(
        containsString("Failed to load pipeline from file"),
        containsString(" - File not found at /foo.txt")
    ));

    evalCommand.pipelineFile = "\\\\foo.txt";
    problem = assertThrows(ExitException.class, () -> evalCommand.run()).getProblem();
    assertThat(render(problem), allOf(
        containsString("Failed to load pipeline from file"),
        containsString(" - File not found at /")
    ));

    evalCommand.pipelineFile = ".\\foo.txt";
    problem = assertThrows(ExitException.class, () -> evalCommand.run()).getProblem();
    assertThat(render(problem), allOf(
        containsString("Failed to load pipeline from file"),
        containsString(" - File not found at /")
    ));
  }

  @Test
  public void canBuildPipelineFromRelativePath() throws Exception {
    evalCommand.pipelineFile = stdhome().resolve("pipeline.txt").toString();
    evalCommand.run();
    assertThat(readCsvColumns("input.csv", "foo"), contains(Lists.newArrayList("bar")));
  }

  @Test
  public void canBuildPipelineFromPath() throws Exception {
    evalCommand.pipelineFile = stdhome().resolve("foo/pipeline.txt").toString();
    evalCommand.run();
    assertThat(readCsvColumns("input.csv", "foo"), contains(Lists.newArrayList("bar")));
  }

  @Test
  public void givesSpellingErrorMessageIfBadString() {
    evalCommand.pipelineFile = "foo * you";
    Problem problem = assertThrows(ExitException.class, () -> evalCommand.run()).getProblem();
    assertThat(render(problem), allOf(
        containsString("Failed to parse pipeline from command-line-argument"),
        containsString("Got '*', wanted"),
        containsString("- To evaluate from a file, prefix the filename")
    ));
  }
}
