/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static org.junit.Assert.*;

import java.nio.file.Files;
import java.nio.file.Path;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;

import org.junit.Test;

import nz.org.riskscape.engine.cli.model.RunCommand;
import nz.org.riskscape.engine.test.EngineCommandIntegrationTest;

public class OutputLocationTest extends EngineCommandIntegrationTest {

  RunCommand runCommand;

  @Test
  public void willUseDefaultOutputLocation() {
    // the output-base-path is relative so will be resolved against the project file location
    runTestModel("project-default-output.ini");
    Path outputBase = stdhome().resolve("output");
    assertResults(outputBase);
  }

  @Test
  public void willUseUserSpecifiedOutputLocation() {
    // the output-base-path is relative so will be resolved against the project file location
    runTestModel("project-user-specified-output.ini");
    Path outputBase = stdhome().resolve("output").resolve("my-output");
    assertResults(outputBase);
  }

  void assertResults(Path outputBase) {
    LocalDateTime now = LocalDateTime.now();
    // we expect the output in a sub-directory of outputBase 'MODEL_ID/TIMESTAMP/results.csv'.
    // the timestamp is at second resolution so we may need to look aback a few seconds to find a match.
    for (int i = 0; i < 10; i++) {
      Path expectedResultPath
          = outputBase.resolve("test").resolve(createDirFriendlyTimestamp(now.minusSeconds(i))).resolve("results.csv");
      if (Files.isReadable(expectedResultPath)) {
        return;
      }
    }
    // we couldn't find the expected file, fail it
    fail("Could not find results at: " +
        outputBase.resolve("test").resolve(createDirFriendlyTimestamp(now)).resolve("results.csv").toString());
  }

  public String createDirFriendlyTimestamp(LocalDateTime dateTime) {
    return DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(dateTime.truncatedTo(ChronoUnit.SECONDS)).replace(":", "_");
  }

  void runTestModel(String projectFile) {
    populateProject(stdhome().resolve(projectFile));

    runCommand = this.setupCommand(new RunCommand());
    runCommand.runnerOptions.replace = true;
    runCommand.modelId = "test";

    runCommand.run();
  }

}
