/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Test;

import nz.org.riskscape.engine.cli.BaseCliIntegrationTest;
import nz.org.riskscape.engine.test.EngineCommandIntegrationTest;

/**
 * Model run integration tests that are run in Docker. This test is a subset of the faster
 * {@link EngineCommandIntegrationTest} so we can be assured they work the same when run in a full RiskScape
 * environment.
 */
public class ModelRunSmokeTests extends BaseCliIntegrationTest {

  /**
   * Use the projectDir (found beneath src/test/homes) as the RiskScape home directory.
   */
  void useProjectHome(String projectDir) throws Exception {
    this.riskscapeDir = Paths.get("src", "test", "homes").resolve(projectDir).toAbsolutePath().toString();
    setupOutputDir();
  }

  @Test
  public void ModelRunPipelineCommandTest_canOverrideModelParametersFromIniFileWithSectionNamedForModelId()
      throws Exception {
    useProjectHome("ModelRunPipelineCommandTest");
    ExecResult result = execute(params.clear().add("--project=project.ini")
        .add("model", "run", "external", "--parameters=parameters.ini", "--output=" + mappedOutputDir.toString()));
    assertThat(result.retCode, is(0));
    // no warnings/errors
    assertThat(result.stderr, hasSize(0));

    List<String> collectedItems = openCsv("sort.csv", "name").stream().map(list -> list.get(0))
        .collect(Collectors.toList());

    assertEquals(Arrays.asList("bee", "dung beetle"), collectedItems);

    // no warnings/errors
    assertThat(result.stderr, hasSize(0));
  }

  @Test
  public void ModelRunPipelineCommandTest_commandLineParametersTakePrecedenceOverModelDefinition() throws Exception {
    useProjectHome("ModelRunPipelineCommandTest");
    ExecResult result = execute(params.clear().add("--project=project.ini")
        .add("model", "run", "inline-with-args", "--parameter=inputOffset=2", "--parameter=inputLimit=1")
        .add("--output=" + mappedOutputDir.toString()));
    assertThat(result.retCode, is(0));
    // no warnings/errors
    assertThat(result.stderr, hasSize(0));

    List<String> collectedItems = openCsv("sort.csv", "name").stream().map(list -> list.get(0))
        .collect(Collectors.toList());

    assertEquals(Arrays.asList("pig"), collectedItems);
  }

  @Test
  public void ModelRunTemplatedCommandTest_canRunAnIncompleteModelBySupplyingMissingParameters() throws Exception {
    useProjectHome("ModelRunPipelineCommandTest");
    ExecResult result = execute(params.clear().add("--project=project.ini")
        .add("model", "run", "inline-param-with-no-default", "--parameter=inputLimit=1")
        .add("--output=" + mappedOutputDir.toString()));
    assertThat(result.retCode, is(0));
    // no warnings/errors
    assertThat(result.stderr, hasSize(0));

    List<String> collectedItems = openCsv("sort.csv", "name").stream().map(list -> list.get(0))
        .collect(Collectors.toList());

    assertEquals(Arrays.asList("dog"), collectedItems);
  }

}
