/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static nz.org.riskscape.engine.Assert.*;
import static nz.org.riskscape.engine.problem.ProblemMatchers.*;
import static org.hamcrest.Matchers.*;
import static org.hamcrest.io.FileMatchers.*;
import static org.junit.Assert.*;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.FileProblems;
import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.engine.cli.ExitException;
import nz.org.riskscape.engine.cli.model.BatchCommand;
import nz.org.riskscape.engine.test.EngineCommandIntegrationTest;

public class ModelBatchCommandTest extends EngineCommandIntegrationTest {

  BatchCommand batchCommand;

  @Before
  public void populateProject() {
    populateProject(stdhome());

    batchCommand = this.setupCommand(new BatchCommand());
    batchCommand.runnerOptions.output = getTempDirectory().toString();
  }

  @Test
  public void canRunBatchVaryingParameter() throws Exception {
    batchCommand.modelId = "read";
    batchCommand.varyParameter = "input";
    batchCommand.varyInputString = stdhome().resolve("batch1.txt").toString();
    batchCommand.run();

    List<List<String>> results = openCsv("output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Standard", "Flying 9", "1937"),
        Arrays.asList("Chevrolet", "Series AA", "1927")
    ));

    results = openCsv("output-1.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Subaru", "Legacy", "1993"),
        Arrays.asList("VW", "Golf", "2014"),
        Arrays.asList("Kia", "Rio", "2021")
    ));
  }

  @Test
  public void canRunBatchVaryingPartOfParameterValue() throws Exception {
    batchCommand.modelId = "read";
    batchCommand.parameters = Arrays.asList("input='{}'");  // we add the parameter with the quoting here
    batchCommand.varyInputString = stdhome().resolve("batch2.txt").toString();
    batchCommand.run();

    List<List<String>> results = openCsv("output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Subaru", "Legacy", "1993"),
        Arrays.asList("VW", "Golf", "2014"),
        Arrays.asList("Kia", "Rio", "2021")
    ));

    results = openCsv("output-1.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Foden", "Alpha", "2004"),
        Arrays.asList("Kenworth", "T659", "2021")
    ));
  }

  @Test
  public void canRunAllFilesInDirectory() throws Exception {
    batchCommand.modelId = "read";
    batchCommand.parameters = Arrays.asList("input=bookmark('{}')");  // we add the parameter with the quoting here
    batchCommand.varyInputString = stdhome().resolve("data").toString();
    batchCommand.run();

    // Batch should be processing batches in filename order,
    // so cars.csv
    List<List<String>> results = openCsv("output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Subaru", "Legacy", "1993"),
        Arrays.asList("VW", "Golf", "2014"),
        Arrays.asList("Kia", "Rio", "2021")
    ));

    // then old-cars.csv
    results = openCsv("output-1.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Standard", "Flying 9", "1937"),
        Arrays.asList("Chevrolet", "Series AA", "1927")
    ));

    // and finally trucks.csv
    results = openCsv("output-2.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Foden", "Alpha", "2004"),
        Arrays.asList("Kenworth", "T659", "2021")
    ));
  }

  @Test
  public void canRunAllFilesInDirectoryMatchingPattern() throws Exception {
    batchCommand.modelId = "read";
    batchCommand.parameters = Arrays.asList("input=bookmark('{}')");
    // set the search to "*ca*csv" which will match all files ending with *cars.csv, but also test that all the
    // wild cards get replaced.
    batchCommand.varyInputString = stdhome().resolve("data").toString() + "/*ca*csv";
    batchCommand.run();

    // Batch should be processing batches in filename order,
    // so cars.csv
    List<List<String>> results = openCsv("output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Subaru", "Legacy", "1993"),
        Arrays.asList("VW", "Golf", "2014"),
        Arrays.asList("Kia", "Rio", "2021")
    ));

    // then old-cars.csv
    results = openCsv("output-1.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Standard", "Flying 9", "1937"),
        Arrays.asList("Chevrolet", "Series AA", "1927")
    ));

    // make sure there isn't a batch for trucks
    assertThat(getTempDirectory().resolve("output-2.csv").toFile(), not(anExistingFile()));
  }

  @Test
  public void canRunBatchSavingResultInBatchDirectory() throws Exception {
    batchCommand.modelId = "read";
    batchCommand.runnerOptions.output = getTempDirectory().resolve("batch-{}").toString();
    batchCommand.parameters = Arrays.asList("input='{}'");  // we add the parameter with the quoting here
    batchCommand.varyInputString = stdhome().resolve("batch2.txt").toString();
    batchCommand.run();

    List<List<String>> results = openCsv("batch-cars/output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Subaru", "Legacy", "1993"),
        Arrays.asList("VW", "Golf", "2014"),
        Arrays.asList("Kia", "Rio", "2021")
    ));

    results = openCsv("batch-trucks/output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Foden", "Alpha", "2004"),
        Arrays.asList("Kenworth", "T659", "2021")
    ));
  }

  @Test
  public void canRunAllFilesInDirectorySavingResultInBatchDirectory() throws Exception {
    batchCommand.modelId = "read";
    batchCommand.runnerOptions.output = getTempDirectory().resolve("batch-{}").toString();
    batchCommand.parameters = Arrays.asList("input=bookmark('{}')");  // we add the parameter with the quoting here
    batchCommand.varyInputString = stdhome().resolve("data").toString();
    batchCommand.run();

    // Batch should be processing batches in filename order,
    // so cars.csv
    List<List<String>> results = openCsv("batch-cars/output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Subaru", "Legacy", "1993"),
        Arrays.asList("VW", "Golf", "2014"),
        Arrays.asList("Kia", "Rio", "2021")
    ));

    // then old-cars.csv
    results = openCsv("batch-old-cars/output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Standard", "Flying 9", "1937"),
        Arrays.asList("Chevrolet", "Series AA", "1927")
    ));

    // and finally trucks.csv
    results = openCsv("batch-trucks/output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Foden", "Alpha", "2004"),
        Arrays.asList("Kenworth", "T659", "2021")
    ));
  }

  @Test
  public void canRunBatchAndVaryParameterFromParametersFile() throws Exception {
    batchCommand.modelId = "read";
    // the parameters file contains:
    // input = 'trucks'
    // limit = {}
    batchCommand.parametersFile = stdhome().resolve("parameters.ini");
    batchCommand.varyInputString = stdhome().resolve("batch-limits.txt").toString();
    batchCommand.run();

    // first batch has limit maxint()
    List<List<String>> results = openCsv("output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Foden", "Alpha", "2004"),
        Arrays.asList("Kenworth", "T659", "2021")
    ));

    // second batch has limit 1
    results = openCsv("output-1.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Foden", "Alpha", "2004")
    ));
  }

  @Test
  public void reallyBadVaryParametersPreventBatchFromRunning() throws Exception {
    batchCommand.modelId = "read";
    batchCommand.varyParameter = "input";
    // this batch is broken because the file contains an entry that cannot be parsed as an expression
    batchCommand.varyInputString = stdhome().resolve("batch-broken.txt").toString();
    ExitException ex = assertThrows(ExitException.class, () -> batchCommand.run());
    String rendered = render(ex.getProblem());
    assertThat(rendered, allOf(
        containsString("Problems found with 'bad-expression"),
        containsString("Got EOF before string was terminated at position 15")
    ));

    // we check to see if any files were created in the output directory. No files means no batches
    // got run.
    assertThat(Files.list(getTempDirectory()).collect(Collectors.toList()), hasSize(0));
  }

  @Test
  public void badBatchesAreWarnedByDefault() throws Exception {
    batchCommand.modelId = "read";
    batchCommand.runnerOptions.output = getTempDirectory().resolve("batch-{}").toString();
    batchCommand.parameters = Arrays.asList("input='{}'");
    batchCommand.varyInputString = stdhome().resolve("batch-bad-entry.txt").toString();
    batchCommand.run();

    String errors = errBytes.toString();
    assertThat(errors, allOf(
        containsString("Could not execute batch 2 of 3"),
        containsString("Vary parameter value: unknown"),
        containsString("No bookmark with id 'unknown' exists")
    ));

    // we should have results for cars
    List<List<String>> results = openCsv("batch-cars/output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Subaru", "Legacy", "1993"),
        Arrays.asList("VW", "Golf", "2014"),
        Arrays.asList("Kia", "Rio", "2021")
    ));

    // and for trucks
    results = openCsv("batch-trucks/output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Foden", "Alpha", "2004"),
        Arrays.asList("Kenworth", "T659", "2021")
    ));
  }

  @Test
  public void badBatchesCanProduceExitException() throws Exception {
    // this mode will run all batches (like default warn) but the command itself will fail if
    // any of the batches failed.
    batchCommand.failureMode = BatchCommand.BatchFailureMode.EXIT_ERROR;
    batchCommand.modelId = "read";
    batchCommand.runnerOptions.output = getTempDirectory().resolve("batch-{}").toString();
    batchCommand.parameters = Arrays.asList("input='{}'");
    batchCommand.varyInputString = stdhome().resolve("batch-bad-entry.txt").toString();
    ExitException ex = assertThrows(ExitException.class, () -> batchCommand.run());

    assertThat(ex.getMessage(), is("There were 1 batch(s) that did not complete successfully"));

    String errors = errBytes.toString();
    assertThat(errors, allOf(
        containsString("Could not execute batch 2 of 3"),
        containsString("Vary parameter value: unknown"),
        containsString("No bookmark with id 'unknown' exists")
    ));

    // we should have results for cars
    List<List<String>> results = openCsv("batch-cars/output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Subaru", "Legacy", "1993"),
        Arrays.asList("VW", "Golf", "2014"),
        Arrays.asList("Kia", "Rio", "2021")
    ));

    // and for trucks
    results = openCsv("batch-trucks/output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Foden", "Alpha", "2004"),
        Arrays.asList("Kenworth", "T659", "2021")
    ));
  }

  @Test
  public void badBatchesCanAbortTheBatchRun() throws Exception {
    // this mode will abort on any failed batch
    batchCommand.failureMode = BatchCommand.BatchFailureMode.ABORT;
    batchCommand.modelId = "read";
    batchCommand.runnerOptions.output = getTempDirectory().resolve("batch-{}").toString();
    batchCommand.parameters = Arrays.asList("input='{}'");
    batchCommand.varyInputString = stdhome().resolve("batch-bad-entry.txt").toString();
    ExitException ex = assertThrows(ExitException.class, () -> batchCommand.run());
    assertThat(ex.getMessage(), is("Batch failed with --vary-parameter value: unknown"));

    String errors = errBytes.toString();
    assertThat(errors, allOf(
        containsString("Could not execute batch 2 of 3"),
        containsString("Vary parameter value: unknown"),
        containsString("No bookmark with id 'unknown' exists, and no supported file with that name could be "
            + "found either")
    ));

    // we should have results for cars
    List<List<String>> results = openCsv("batch-cars/output.csv", "make", "model", "year");
    assertThat(results, contains(
        Arrays.asList("Subaru", "Legacy", "1993"),
        Arrays.asList("VW", "Golf", "2014"),
        Arrays.asList("Kia", "Rio", "2021")
    ));

    // trucks should not exist because we aborted before we got there
    Path trucks = getTempDirectory().resolve("batch-trucks");
    assertFalse(trucks.toFile().exists());
  }

  @Test
  public void badOutputPathAbortsTheRun() {
    batchCommand.modelId = "read";
    // The < is not allowed in a URI, this should stop the command from running before any batches are started
    batchCommand.runnerOptions.output = String.format("%s/<my-results>", getTempDirectory().toString());
    batchCommand.parameters = Arrays.asList("input='{}'");  // we add the parameter with the quoting here
    batchCommand.varyInputString = stdhome().resolve("batch2.txt").toString();
    ExitException ex = assertThrows(ExitException.class, () -> batchCommand.run());
    assertThat(ex.getProblem(), Matchers.hasAncestorProblem(isProblem(FileProblems.class, (r, p) -> p.badUri(
        r.eq(batchCommand.runnerOptions.output),
        r.match(containsString("Illegal character in path at index"))
    ))));
  }

  @Test
  public void badOutputPathFromVaryParamAbortsTheBatch() {
    batchCommand.modelId = "greet";
    // The < is not allowed in a URI, this should stop the command from running before any batches are started
    batchCommand.runnerOptions.output = String.format("%s/{}", getTempDirectory().toString());
    batchCommand.parameters = Arrays.asList("name='{}'");  // we add the parameter with the quoting here
    batchCommand.varyInputString = stdhome().resolve("names.txt").toString();

    batchCommand.run();

    String errors = errBytes.toString();
    String output = outBytes.toString();

    assertThat(output, containsString("Batch complete. Success 3 of 4"));
    assertThat(errors, allOf(
        containsString("Could not execute batch 3 of 4"),
        containsString("Vary parameter value: foo<bar"),
        containsString("foo<bar' is a bad URI - Illegal character in path at index")
    ));
  }

}
