/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;

import org.junit.Test;


public class IntermediateModellingTutorialTest extends BaseModelRunCommandTest {
  @Override
  public Path stdhome() {
    return Paths.get("..", "..", "examples", "intermediate-modelling");
  }

  @Test
  public void  canRunSampleCentroidModel() throws Exception {
    runCommand.modelId = "sample-centroid";
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Results","Number_Exposed", "Total_buildings",
        "Max_hazard", "Average_hazard");

    assertThat(results, contains(Arrays.asList("Total","1483", "6260", "5.786825180053711",
        "1.7566953226270823")));
  }

  @Test
  public void  canRunSampleClosestModel() throws Exception {
    runCommand.modelId = "sample-closest";
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Results","Number_Exposed", "Total_buildings",
        "Max_hazard", "Average_hazard");

    assertThat(results, contains(Arrays.asList("Total","1568", "6260", "5.786825180053711", "1.6731144939454234")));
  }

  @Test
  public void  canRunSampleAllIntersectionsModel() throws Exception {
    runCommand.modelId = "sample-all-intersections";
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Results","Number_Exposed", "Total_buildings",
        "Max_hazard", "Average_hazard");

    assertThat(results, contains(Arrays.asList("Total","1568", "6260", "7.131937503814697", "1.9376521151208757")));
  }

  @Test
  public void  canRunSampleAllIntersectionsModelAndSaveRawResults() throws Exception {
    runCommand.modelId = "sample-all-intersections";
    runCommand.runnerOptions.format = "csv";
    runCommand.parameters = Arrays.asList("analysis.save-raw-results=true");

    runCommand.doCommand(project);

    List<List<String>> results = openCsv("raw-analysis.csv", "hazard_sampled.geometry","hazard_sampled.sampled",
        "consequence","exposed_ratio","exposure.the_geom","exposure.Use_Cat","exposure.Cons_Frame",
        "exposure.area","exposure.perimeter","hazard");

    assertThat(results, hasItem(Arrays.asList("POLYGON ((423955 8448425, 423950.15866954095 8448425,"
        + " 423950.247885206 8448429.76745714, 423955 8448429.634136112, 423955 8448425))",
        "0.9268975257873535", "1", "0.15947752494724476",
        "MULTIPOLYGON (((423950.06159462 8448419.81256431, 423950.247885206 8448429.76745714, 423964.502861344"
        + " 8448429.36753244, 423964.531870331 8448419.63448918, 423950.06159462 8448419.81256431)))",
        "Residential", "Masonry", "141.407958984375", "48.42167862604659", "0.93930983543396")
    ));
  }

  @Test
  public void  CanRunSampleClosestModelwithBuffer() throws Exception {
    runCommand.modelId = "sample-closest";
    runCommand.parameters = Arrays.asList("sample.hazards-buffer=1");

    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Results","Number_Exposed", "Total_buildings",
        "Max_hazard", "Average_hazard");

    assertThat(results, hasItem(Arrays.asList("Total","1580", "6260","5.786825180053711",
        "1.6614069632337063")
    ));
  }

  @Test
  public void  CanRunSampleAllIntersectionsModelwithBuffer() throws Exception {
    runCommand.modelId = "sample-all-intersections";
    runCommand.parameters = Arrays.asList("sample.hazards-buffer=1");

    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Results","Number_Exposed", "Total_buildings",
        "Max_hazard", "Average_hazard");

    assertThat(results, hasItem(Arrays.asList("Total","1580", "6260","7.141936302185059",
        "1.9916315742685824")
    ));
  }

  @Test
  public void  CanRunAggregateHazardModel() throws Exception {
    runCommand.modelId = "aggregate-hazard";
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Results","Number_Exposed","DS5_Percentile.P25",
        "DS5_Percentile.P50","DS5_Percentile.P75");

    assertThat(results, hasItem(Arrays.asList("Total","1568","0.03664936501736071","0.5600526961725251",
        "0.955723150400208")
    ));
  }

  @Test
  public void  CanRunAggregateByConsequenceModel() throws Exception {
    runCommand.modelId = "aggregate-consequence";
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Results","Number_Exposed","DS5_Percentile.P25",
        "DS5_Percentile.P50","DS5_Percentile.P75");

    assertThat(results, hasItem(Arrays.asList("Total","1568","0.056198939562588174",
        "0.544483500859628","0.9462498383368769")
    ));
  }

  @Test
  public void  CheckThatAggregateModelsProduceSameOutput() throws Exception {
    runCommand.modelId = "aggregate-consequence";
    runCommand.parameters = Arrays.asList("analysis.aggregate-hazards-function=max");
    runCommand.doCommand(project);

    List<List<String>> resultsConsequence = openCsv("event-impact.csv","Results","Number_Exposed","DS5_Percentile.P25",
        "DS5_Percentile.P50","DS5_Percentile.P75");

    runCommand.modelId = "aggregate-hazard";
    runCommand.parameters = Arrays.asList("analysis.aggregate-consequences-function=max");
    runCommand.doCommand(project);

    List<List<String>> resultsHazard = openCsv("event-impact.csv","Results","Number_Exposed","DS5_Percentile.P25",
        "DS5_Percentile.P50","DS5_Percentile.P75");

    assertThat(resultsHazard, equalTo(resultsConsequence));
  }
}
