/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;

import org.junit.Test;

import nz.org.riskscape.engine.FileSystemMatchers;

public class GeoprocessingTutorialTest extends BaseModelRunCommandTest {

  @Override
  public Path stdhome() {
    return Paths.get("..", "..", "examples", "geoprocessing-tutorial");
  }

  @Test
  public void canRunSimpleExposureModel() throws Exception {
    runCommand.modelId = "simple-exposure";
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Region", "Exposed_segments", "Exposed_km");

    assertThat(results, contains(
        Arrays.asList("Aleipata Itupa i Lalo", "24", "24.287"),
        Arrays.asList("Aleipata Itupa i Luga", "30", "19.661"),
        Arrays.asList("Falealili", "67", "42.263"),
        Arrays.asList("Lepa", "10", "7.69"),
        Arrays.asList("Lotofaga", "41", "16.58")
    ));
  }

  @Test
  public void canRunCutByDistanceModel() throws Exception {
    runCommand.modelId = "cut-by-distance";
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Region", "Exposed_segments", "Exposed_km");

    assertThat(results, contains(
        Arrays.asList("Aleipata Itupa i Lalo", "200", "9.558"),
        Arrays.asList("Aleipata Itupa i Luga", "237", "11.301"),
        Arrays.asList("Falealili", "347", "15.936"),
        Arrays.asList("Lepa", "116", "5.588"),
        Arrays.asList("Lotofaga", "108", "4.138")
    ));
  }

  @Test
  public void canRunExposedLengthModel() throws Exception {
    runCommand.modelId = "exposed-length";
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Region", "Exposed_segments", "Exposed_km");

    assertThat(results, contains(
        Arrays.asList("Aleipata Itupa i Lalo", "24", "10.661"),
        Arrays.asList("Aleipata Itupa i Luga", "30", "12.333"),
        Arrays.asList("Falealili", "67", "14.088"),
        Arrays.asList("Lepa", "10", "0.902"),
        Arrays.asList("Lotofaga", "41", "5.489")
    ));
  }

  @Test
  public void canRunCutByLayerModel() throws Exception {
    runCommand.modelId = "cut-by-layer";
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Region", "Exposed_segments", "Exposed_km");

    assertThat(results, contains(
        Arrays.asList("Aleipata Itupa i Lalo", "25", "9.216"),
        Arrays.asList("Aleipata Itupa i Luga", "37", "10.614"),
        Arrays.asList("Falealili", "68", "14.799"),
        Arrays.asList("Lepa", "12", "5.225"),
        Arrays.asList("Lotofaga", "42", "3.62")
    ));
  }

  @Test
  public void canRunFilterByRegionModel() throws Exception {
    runCommand.modelId = "filter-by-region";
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Region", "Exposed_segments", "Exposed_km");

    assertThat(results, contains(
        Arrays.asList("Falealili", "67", "14.088"),
        Arrays.asList("Lotofaga", "1", "1.328")
    ));
  }

  @Test
  public void canRunIncludeOnlyRegionModel() throws Exception {
    runCommand.modelId = "include-only-region";
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Region", "Exposed_segments", "Exposed_km");

    assertThat(results, contains(
        Arrays.asList("Falealili", "68", "14.799")
    ));
  }

  @Test
  public void canRunBufferRoadsModel() throws Exception {
    runCommand.modelId = "buffer-roads";
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Region", "Exposed_segments", "Exposed_km", "Exposed_m2");

    assertThat(results, contains(
        Arrays.asList("Aleipata Itupa i Lalo", "25", "24.427", "30515"),
        Arrays.asList("Aleipata Itupa i Luga", "30", "19.661", "32572"),
        Arrays.asList("Falealili", "67", "42.263", "36775"),
        Arrays.asList("Lepa", "10", "7.69", "2277"),
        Arrays.asList("Lotofaga", "41", "16.58", "11937")
    ));
  }

  @Test
  public void canRunBufferNoOverlapModel() throws Exception {
    runCommand.modelId = "buffer-no-overlap";
    // sanity-check the sampled geometry can be saved to file (i.e. no mismatching types)
    runCommand.parameters = Arrays.asList("analysis.save-raw-results=true");
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Region", "Exposed_segments", "Exposed_km", "Exposed_m2");

    assertThat(results, contains(
        Arrays.asList("Aleipata Itupa i Lalo", "25", "24.427", "30452"),
        Arrays.asList("Aleipata Itupa i Luga", "30", "19.661", "32528"),
        Arrays.asList("Falealili", "67", "42.263", "36657"),
        Arrays.asList("Lepa", "10", "7.69", "2245"),
        Arrays.asList("Lotofaga", "41", "16.58", "11918")
    ));

    assertThat(getTempDirectory(), FileSystemMatchers.hasFile(FileSystemMatchers.fileWithName("raw-analysis.shp")));
  }
}
