/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.engine.cli.tests;

import static org.junit.Assert.*;

import java.nio.file.Files;
import java.util.Arrays;
import java.util.List;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.FileSystemMatchers;
import nz.org.riskscape.engine.test.EngineTestSettings;

@EngineTestSettings("engine.csv-include-bom=false") // the bom gets in the way of the file comparison
public class GeoJSONTest extends BaseModelRunCommandTest {

  @Before
  public void runSingleThreaded() {
    engine.getPipelineExecutor().setNumThreads(1);
  }

  @Test
  public void canWritePointsToGeoJSON() throws Exception {
    testGeoJSONroundTrip("points");
  }

  @Test
  public void canWriteLinesToGeoJSON() throws Exception {
    testGeoJSONroundTrip("lines");
  }

  @Test
  public void canWritePolygonsToGeoJSON() throws Exception {
    testGeoJSONroundTrip("polygons");
  }

  private void testGeoJSONroundTrip(String start) throws Exception {
    // First we read a CSV file and save it to geojson
    runCommand.modelId = "read";
    runCommand.parameters = Arrays.asList(String.format("input='%s'", start),
        "format='geojson'");
    runCommand.run();
    assertThat(getTempDirectory(), FileSystemMatchers.hasFile(FileSystemMatchers.fileWithName("output.geojson")));
    List<String> lines = Files.readAllLines(getTempDirectory().resolve("output.geojson"));

    // Now we read the geojson output and save it again to geojson. Then we should have two identical geojson files
    reset();
    runCommand.parameters = Arrays.asList(String.format("input=bookmark('%s')",
        getTempDirectory().resolve("output.geojson").toAbsolutePath()));
    runCommand.run();

    assertThat(getTempDirectory(), FileSystemMatchers.hasFile(FileSystemMatchers.fileWithName("output-1.geojson")));
    List<String> lines1 = Files.readAllLines(getTempDirectory().resolve("output-1.geojson"));

    assertEquals(lines, lines1);

    // we read the geojson file and save it to CSV. The output should be identical to what we started with.
    reset();
    runCommand.parameters = Arrays.asList(String.format("input=bookmark('%s')",
        getTempDirectory().resolve("output.geojson").toAbsolutePath()),
        "format='csv'");
    runCommand.run();
    assertThat(getTempDirectory(), FileSystemMatchers.hasFile(FileSystemMatchers.fileWithName("output.csv")));
    List<String> original = Files.readAllLines(stdhome().resolve(start + ".csv"));
    List<String> read = Files.readAllLines(getTempDirectory().resolve("output.csv"));
    assertEquals(original, read);
  }

}
