/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import org.junit.Test;

import com.opencsv.exceptions.CsvValidationException;

public class GL593_GridExposureLayer extends BaseModelRunCommandTest {

  @Test
  public void canUseAGeotiffAsAnExposureLayerWithAPipeline() throws CsvValidationException, IOException {
    // this is a fairly chunky model, we need muchos threads
    engine.getPipelineExecutor().setNumThreads(Runtime.getRuntime().availableProcessors());
    runCommand.modelId = "population-exposure";
    runCommand.parameters.add("input='western-samoa-popdensity'");
    runCommand.run();

    List<List<String>> rows = openCsv("totals.csv", "exposed_population", "total_population");

    assertThat(rows, containsInAnyOrder(
        Arrays.asList("2651","197305")
    ));

    List<List<String>> regionalResults = openCsv("regions.csv", "name", "exposed_population", "total_population");

    // Spot check some results
    assertThat(regionalResults, allOf(
        hasItem(Arrays.asList("Aana Alofi II", "0", "3725")),
        hasItem(Arrays.asList("Aana Alofi III", "38", "5640")),
        hasItem(Arrays.asList("Falelatai & Samatau", "183", "2700")),
        hasItem(Arrays.asList("Palauli West", "170", "3035")),
        hasItem(Arrays.asList("Sagaga le Falefa", "0", "14806"))
    ));
  }

  @Test
  public void canUseAGeotiffAsAnExposureLayerWithWizard() throws CsvValidationException, IOException {
    // this is a fairly chunky model, we need muchos threads
    engine.getPipelineExecutor().setNumThreads(Runtime.getRuntime().availableProcessors());
    runCommand.modelId = "population-exposure-wizard";
    runCommand.run();

    List<List<String>> regionalResults = openCsv("event-impact.csv", "name", "exposed_population", "total_population");

    // Spot check some results
    assertThat(regionalResults, allOf(
        hasItem(Arrays.asList("Aana Alofi II", "0", "3725")),
        hasItem(Arrays.asList("Aana Alofi III", "38", "5640")),
        hasItem(Arrays.asList("Falelatai & Samatau", "183", "2700")),
        hasItem(Arrays.asList("Palauli West", "170", "3035")),
        hasItem(Arrays.asList("Sagaga le Falefa", "0", "14806"))
    ));
  }

  @Test
  public void canUseAGeotiffWithBookmarkExpression_GL1251() throws CsvValidationException, IOException {
    populateProjectAndSetupCommands(homes().resolve("ModelRunWizardCommandTest"));
    setCommandInput("input(bookmark('kaiju-fire-grid.tif')) -> group({count(*) as total})");
    evalCommand.pipelineFile = "-";
    evalCommand.run();

    String total = openCsv("group.csv", "total").get(0).get(0);
    assertEquals(35, Integer.parseInt(total));
  }

}
