/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.List;
import java.util.Set;

import org.junit.Test;

public class GL472_CombineCoverages extends BaseModelRunCommandTest {

  @Test
  public void canSampleCombinedHazardLayersWithLargeGrid() throws Exception {
    runCommand.modelId = "combine-coverages";
    // cut the roads with a 10km grid, which allows us to manually count the segments affected.
    // Just use the unbuffered road line to start with, as it's easier to see the length exposed
    runCommand.parameters = Arrays.asList("grid_distance=10000", "road_geom=exposure.the_geom");
    runCommand.run();

    Set<List<String>> rows = openCsvUniqueData("grouped.csv", "exposure_id", "magma_squares", "pumice_squares",
        "total_road", "exposed_road");

    // note that:
    // - all of road #2 was exposed to the magma
    // - although road #3 was exposed to magma, this grid size is too big to pick that up
    // - pumice is too small with this grid-size to get many hits
    // - we cut roads by a 10km x 10km grid, but the actual road inside that grid may be
    //   more or less than 10km, due to the way roads curve
    assertThat(
      rows,
      containsInAnyOrder(
          hasItems("1", "9", "1", "105032", "74058"),
          hasItems("2", "2", "1", "11480", "11480")
      )
    );
  }

  @Test
  public void canCombineSomeHazardLayersAndSampleThemAsOne() throws Exception {
    runCommand.modelId = "combine-coverages";
    runCommand.parameters = Arrays.asList("hazard_coverage=pumice_magma");
    runCommand.run();

    Set<List<String>> rows = openCsvUniqueData("grouped.csv", "exposure_id", "magma_squares", "pumice_squares",
        "total_road", "exposed_road");

    assertThat(
      rows,
      containsInAnyOrder(
          // these were 'inspected' against opening up the outputs in QGis and checking they looked ok
          hasItems("1", "1019", "74", "1930450", "1327800"),
          hasItems("2", "178", "71", "203265", "203265"),
          hasItems("3", "33", "0", "239434", "40330")
      )
    );
  }

  @Test
  public void canCombineHazardLayersInDifferentOrderAndGetSameResults() throws Exception {
    runCommand.modelId = "combine-coverages";
    // switch the multicoverage so the coverages are in a different order this time
    runCommand.parameters = Arrays.asList("hazard_coverage=magma_pumice");
    runCommand.run();

    Set<List<String>> rows = openCsvUniqueData("grouped.csv", "exposure_id", "magma_squares", "pumice_squares",
        "total_road", "exposed_road");

    // check we get the same/similar results
    // (the segment count is slightly different as the grid now starts in a different place)
    assertThat(
      rows,
      containsInAnyOrder(
          hasItems("1", "1027", "74", "1930450", "1326646"),
          hasItems("2", "170", "70", "203265", "203265"),
          hasItems("3", "34", "0", "239434", "40300")
      )
    );
  }
}
