/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static org.hamcrest.MatcherAssert.*;
import static org.hamcrest.Matchers.*;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

import org.hamcrest.Matcher;
import org.junit.Test;
import org.junit.experimental.categories.Category;

import nz.org.riskscape.engine.cli.BaseCliIntegrationTest;
import nz.org.riskscape.engine.cli.CliIntegrationTest;
import nz.org.riskscape.engine.cli.RunParameters;

@Category(CliIntegrationTest.class)
public class GL1365_Beta_Switches extends BaseCliIntegrationTest {

  @Test
  public void byDefaultBetaIsDisabled() throws Exception {
    ExecResult result = execute(params.clear().add("--version"));

    assertThat(
        result,
        allOf(
            hasPluginList(not(betaPlugin())),
            not(hasBetaPluginWarning())
        )
    );
  }

  @Test
  public void canBeEnabledWithBetaSwitch() throws Exception {
    ExecResult result = execute(params.clear().add("--version").add("--beta"));

    assertThat(
        result,
        allOf(
            hasPluginList(betaPlugin()),
            hasBetaPluginWarning()
        )
    );
  }

  @Test
  public void canBeEnabledViaSettings() throws Exception {
    ExecResult result = execute(addSettingsHomeDir(params.clear().add("--version")));

    assertThat(
        result,
        allOf(
            hasPluginList(betaPlugin()),
            hasBetaPluginWarning()
        )
    );
  }

  @Test
  public void settingCanBeOverriddenWithNoBetaSwitch() throws Exception {
    ExecResult result = execute(addSettingsHomeDir(params.clear().add("--version").add("--no-beta")));

    assertThat(
        result,
        allOf(
            hasPluginList(not(betaPlugin())),
            not(hasBetaPluginWarning())
        )
    );
  }

  /**
   * Write out settings file and point riskscape at it
   */
  private RunParameters addSettingsHomeDir(RunParameters params) throws IOException {
    String settings = """
        [global]
        load-plugin = beta
        """;

    Path settingsPath = outputDir.resolve("settings.ini");
    Files.write(settingsPath, settings.getBytes());

    return params.add("--home").add(mappedOutputDir.toString());
  }

  /**
   * Confirm that stderr contains the beta plugin warning
   */
  private Matcher<ExecResult> hasBetaPluginWarning() {
    return hasProperty("stderr", hasItem(containsString("[WARNING] The 'beta' plugin is enabled")));
  }

  /**
   * Confirm that stdout is showing the plugin listing
   */
  private Matcher<ExecResult> hasPluginList(Matcher<String> pluginNameMatcher) {
    return hasProperty(
        "stdout",
        hasItems(containsString("Plugins"), containsString("EnginePlugin"), pluginNameMatcher)
    );
  }

  /**
   * Match output that shows part of the beta plugin class name
   */
  private Matcher<String> betaPlugin() {
    return containsString("beta.Plugin");
  }
}
