/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static org.junit.Assert.assertEquals;

import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.io.ByteOrderMark;
import org.apache.commons.io.input.BOMInputStream;
import org.junit.Test;

public class GL1247_UTF_BOM extends BaseModelRunCommandTest {

    public static final ByteOrderMark[] ALL_BOMS = {
            ByteOrderMark.UTF_8,
            ByteOrderMark.UTF_16LE,
            ByteOrderMark.UTF_16BE,
            ByteOrderMark.UTF_32LE,
            ByteOrderMark.UTF_32BE
    };

    private void assertBom(Path path, ByteOrderMark expected) throws Exception {
        InputStream in = Files.newInputStream(path);
        BOMInputStream bomIn = new BOMInputStream(in, ALL_BOMS);
        assertEquals(expected, bomIn.getBOM());
    }

    private void assertBom(String filename, ByteOrderMark expected) throws Exception {
        assertBom(stdhome().resolve(filename), expected);
    }

    private void assertOutputUtf8Bom() throws Exception {
        assertBom(getTempDirectory().resolve("select.csv"), ByteOrderMark.UTF_8);
    }

    @Test
    public void canReadUTF8WithBOM() throws Exception {

        // Make sure that the file is actually UTF-8 and has a BOM
        assertBom("bom8.csv", ByteOrderMark.UTF_8);

        runCommand.modelId = "test";
        runCommand.parameters.add("input='bom8.csv'");
        runCommand.run();

        List<List<String>> endianness = openCsv("select.csv", "Endianness");

        List<List<String>> expected = Arrays.asList(
                Arrays.asList("N/A"),
                Arrays.asList("Little"),
                Arrays.asList("Big"),
                Arrays.asList("Little"),
                Arrays.asList("Big")
        );

        assertEquals(expected, endianness);
        assertOutputUtf8Bom();
    }


    @Test
    public void canReadUTF16LEWithBOM() throws Exception {

        // Make sure that the file is actually UTF-16 and has a BOM
        assertBom("bom16LE.csv", ByteOrderMark.UTF_16LE);

        runCommand.modelId = "test";
        runCommand.parameters.add("input='bom16LE.csv'");
        runCommand.run();

        List<List<String>> endianness = openCsv("select.csv", "Endianness");

        List<List<String>> expected = Arrays.asList(
                Arrays.asList("N/A"),
                Arrays.asList("Little"),
                Arrays.asList("Big"),
                Arrays.asList("Little"),
                Arrays.asList("Big")
        );

        assertEquals(expected, endianness);
        assertOutputUtf8Bom();
    }

    @Test
    public void canReadUTF16BEWithBOM() throws Exception {

        // Make sure that the file is actually UTF-16 and has a BOM
        assertBom("bom16BE.csv", ByteOrderMark.UTF_16BE);

        runCommand.modelId = "test";
        runCommand.parameters.add("input='bom16BE.csv'");
        runCommand.run();

        List<List<String>> endianness = openCsv("select.csv", "Endianness");

        List<List<String>> expected = Arrays.asList(
                Arrays.asList("N/A"),
                Arrays.asList("Little"),
                Arrays.asList("Big"),
                Arrays.asList("Little"),
                Arrays.asList("Big")
        );

        assertEquals(expected, endianness);
        assertOutputUtf8Bom();
    }

    @Test
    public void canReadUTF32LEWithBOM() throws Exception {

        // Make sure that the file is actually UTF-32 and has a BOM
        assertBom("bom32LE.csv", ByteOrderMark.UTF_32LE);

        runCommand.modelId = "test";
        runCommand.parameters.add("input='bom32LE.csv'");
        runCommand.run();

        List<List<String>> endianness = openCsv("select.csv", "Endianness");

        List<List<String>> expected = Arrays.asList(
                Arrays.asList("N/A"),
                Arrays.asList("Little"),
                Arrays.asList("Big"),
                Arrays.asList("Little"),
                Arrays.asList("Big")
        );

        assertEquals(expected, endianness);
        assertOutputUtf8Bom();
    }

    @Test
    public void canReadUTF32BEWithBOM() throws Exception {

        // Make sure that the file is actually UTF-32 and has a BOM
        assertBom("bom32BE.csv", ByteOrderMark.UTF_32BE);

        runCommand.modelId = "test";
        runCommand.parameters.add("input='bom32BE.csv'");
        runCommand.run();

        List<List<String>> endianness = openCsv("select.csv", "Endianness");

        List<List<String>> expected = Arrays.asList(
                Arrays.asList("N/A"),
                Arrays.asList("Little"),
                Arrays.asList("Big"),
                Arrays.asList("Little"),
                Arrays.asList("Big")
        );

        assertEquals(expected, endianness);
        assertOutputUtf8Bom();
    }
}
