/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static nz.org.riskscape.engine.Matchers.*;
import static org.junit.Assert.*;

import java.io.File;
import java.net.URI;
import java.util.Arrays;
import java.util.List;

import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.FileProblems;
import nz.org.riskscape.engine.cli.ExitException;
import nz.org.riskscape.engine.defaults.data.WfsBookmarkResolver;
import org.junit.Test;

/**
 * Checks we get decent errors when reading dud files, i.e. make it obvious it's the file that has
 * the problem, not RiskScape.
 */
public class DudFileErrorHandlingTest extends BaseModelRunCommandTest {

  public static final List<String> FILE_TYPES = Arrays.asList("shp", "gpkg", "geojson", "tiff");

  @Test
  public void sensibleErrorWhenReadingInvalidFile() {
    // should get a sensible error message if we try to use a blatantly bad file
    runCommand.modelId = "read-input";
    for (String format : FILE_TYPES) {
      File file = stdhome().resolve("bad." + format).toAbsolutePath().toFile();
      runCommand.parameters = Arrays.asList("bookmark=bad." + format);
      ExitException ex = Assert.assertThrows(ExitException.class, () -> runCommand.run());
      assertThat(ex.getProblem(), hasAncestorProblem(equalIgnoringChildren(
              FileProblems.get().geotoolsCannotRead(file)
      )));
    }
  }

  @Test
  public void sensibleErrorWhenReadingFileInWrongFormat() {
    // same thing, this time simulating a valid file being saved with the wrong file extension
    runCommand.modelId = "read-input";
    for (String format : FILE_TYPES) {
      File file = stdhome().resolve("wrong-format." + format).toAbsolutePath().toFile();
      runCommand.parameters = Arrays.asList("bookmark=wrong-format." + format);
      ExitException ex = Assert.assertThrows(ExitException.class, () -> runCommand.run());
      assertThat(ex.getProblem(), hasAncestorProblem(equalIgnoringChildren(
              FileProblems.get().geotoolsCannotRead(file)
      )));
    }
  }

  @Test
  public void sensibleErrorForBadWfslink() throws Exception {
    // check a bad WFS link - we get a slightly different error message for that case
    runCommand.modelId = "read-input";
    runCommand.parameters = Arrays.asList("bookmark=bad-wfs");
    ExitException ex = Assert.assertThrows(ExitException.class, () -> runCommand.run());
    assertThat(ex.getProblem(), hasAncestorProblem(equalIgnoringChildren(
            WfsBookmarkResolver.PROBLEMS.badWfsTarget(new URI("https://riskscape.nz/not-a-wfs-endpoint"))
    )));
  }
}
