/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.List;

import org.junit.Test;

import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.RowMatchers;
import nz.org.riskscape.engine.cli.ExitException;

public class CurveFittingTests extends BaseModelRunCommandTest {

  @Test
  public void canFitAPowerLawCurve() throws Exception {
    runCommand.runnerOptions.replace = true;
    runCommand.modelId = "fit";
    runCommand.run();

    List<List<String>> losses = openCsv("losses.csv", "aep", "interpolated_loss", "fitted");

    assertThat(
      losses,
      containsInAnyOrder(
        contains("1.0", "140", "powerlaw"),
        contains("0.75", "172", "powerlaw"),
        contains("0.5", "230", "powerlaw"),
        contains("0.25", "378", "powerlaw"),
        contains("0.1", "730", "powerlaw"),
        contains("0.05", "1200", "powerlaw"),
        contains("0.01", "3808", "powerlaw")
      )
    );
  }

  @Test
  public void canFitALinearCurve() throws Exception {
    runCommand.runnerOptions.replace = true;
    runCommand.modelId = "fit";
    runCommand.parameters = Arrays.asList("losses='linear-losses.csv'");
    runCommand.run();

    List<List<String>> losses = openCsv("losses.csv", "aep", "interpolated_loss", "fitted");
    double tolerance = 1.1;

    assertThat(
      losses,
      containsInAnyOrder(
        contains(equalTo("1.0"), RowMatchers.numberWithDelta(20, tolerance), equalTo("linear")),
        contains(equalTo("0.75"), RowMatchers.numberWithDelta(45, tolerance), equalTo("linear")),
        contains(equalTo("0.5"), RowMatchers.numberWithDelta(70, tolerance), equalTo("linear")),
        contains(equalTo("0.25"), RowMatchers.numberWithDelta(95, tolerance), equalTo("linear")),
        contains(equalTo("0.1"), RowMatchers.numberWithDelta(110, tolerance), equalTo("linear")),
        contains(equalTo("0.05"), RowMatchers.numberWithDelta(115, tolerance), equalTo("linear")),
        contains(equalTo("0.01"), RowMatchers.numberWithDelta(119, tolerance), equalTo("linear"))
      )
    );
  }

  @Test
  public void canFitAContinuousCurve() throws Exception {
    runCommand.runnerOptions.replace = true;
    runCommand.modelId = "fit";
    runCommand.parameters = Arrays.asList(
      "fitters={linear: {}, continuous: {score: maxfloat()}}",
      "aeps=[1.1, 1.0, 0.01, 0.001]"
    );

    runCommand.run();

    List<List<String>> losses = openCsv("losses.csv", "aep", "interpolated_loss", "fitted");

    // leave the specifics of checking the continuous function to the unit tests
    assertThat(
      losses,
      containsInAnyOrder(
        contains("1.1", "100", "continuous"),
        contains("1.0", "100", "continuous"),
        contains("0.01", "3000", "continuous"),
        contains("0.001", "3000", "continuous")
      )
    );
  }

  @Test
  public void canFitAContinuousCurveIgnoringExtraneousOptions() throws Exception {
    // currently any extraneous options are ignored. Maybe this should be an error.
    runCommand.runnerOptions.replace = true;
    runCommand.modelId = "fit";
    runCommand.parameters = Arrays.asList(
      "fitters={linear: {}, continuous: {score: maxfloat(), bogus: 'foo'}}",
      "aeps=[1.1, 1.0, 0.01, 0.001]"
    );

    runCommand.run();

    List<List<String>> losses = openCsv("losses.csv", "aep", "interpolated_loss", "fitted");

    // leave the specifics of checking the continuous function to the unit tests
    assertThat(
      losses,
      containsInAnyOrder(
        contains("1.1", "100", "continuous"),
        contains("1.0", "100", "continuous"),
        contains("0.01", "3000", "continuous"),
        contains("0.001", "3000", "continuous")
      )
    );
  }

  @Test
  public void failsIfFitterOptionsDoNotBind() throws Exception {
    runCommand.runnerOptions.replace = true;
    runCommand.modelId = "fit";
    runCommand.parameters = Arrays.asList(
      "fitters={linear: {}, continuous: {score: 'awesome'}}",
      "aeps=[1.1, 1.0, 0.01, 0.001]"
    );

    ExitException ex = Assert.assertThrows(ExitException.class, () -> runCommand.run());
    assertThat(
        render(ex.getProblem()),
        containsString("Could not convert 'awesome' into a number (double)")
    );
  }


}
