/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;

import org.junit.Before;
import org.junit.Test;


public class CreatingAProjectTutorialTest extends BaseModelRunCommandTest {
  @Override
  public Path stdhome() {
    return Paths.get("..", "..", "examples", "project-tutorial");
  }

  @Test
  public void canRunBasicExposureModel() throws Exception {
    runCommand.modelId = "exposure-by-region";
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Region","Exposed_buildings");

    assertThat(results, contains(Arrays.asList("","10"),
        Arrays.asList("Aleipata Itupa i Lalo","526"),
        Arrays.asList("Aleipata Itupa i Luga","340"),
        Arrays.asList("Falealili","749"),
        Arrays.asList("Lepa","288"),
        Arrays.asList("Lotofaga","146")
    ));
  }

  @Test
  public void  canRunExposureByRegionandUseSamoaDistrictsBookmarkParam() throws Exception {
    runCommand.modelId = "exposure-by-region";
    runCommand.parameters = Arrays.asList("input-areas.layer=Samoa_districts");
    runCommand.doCommand(project);
    List<List<String>> results = openCsv("event-impact.csv","Region","Exposed_buildings");

    assertThat(results, contains(
        Arrays.asList("","19"),
        Arrays.asList("Aleipata Itupa i Lalo","518"),
        Arrays.asList("Aleipata Itupa i Luga","341"),
        Arrays.asList("Falealili","749"),
        Arrays.asList("Lepa","286"),
        Arrays.asList("Lotofaga","146")
    ));
  }

  @Test
  public void  canRunExposureByRegionUsingBuldingCentriodsCsvBookmark() throws Exception {
    runCommand.modelId = "exposure-by-region";
    runCommand.parameters = Arrays.asList("input-exposures.layer=building_centroids_csv");
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Region","Exposed_buildings");

    assertThat(results, contains(
        Arrays.asList("","10"),
        Arrays.asList("Aleipata Itupa i Lalo","516"),
        Arrays.asList("Aleipata Itupa i Luga","329"),
        Arrays.asList("Falealili","723"),
        Arrays.asList("Lepa","276"),
        Arrays.asList("Lotofaga","141")
    ));
  }

  @Test
  public void  canRunExposureByRegionWithThresholdFunction() throws Exception {
    runCommand.modelId = "exposure-by-region";
    runCommand.parameters = Arrays.asList("analysis.function=exceeds_threshold");
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Region","Exposed_buildings");

    assertThat(results, contains(
        Arrays.asList("","10"),
        Arrays.asList("Aleipata Itupa i Lalo","498"),
        Arrays.asList("Aleipata Itupa i Luga","318"),
        Arrays.asList("Falealili","704"),
        Arrays.asList("Lepa","264"),
        Arrays.asList("Lotofaga","138")
    ));
  }

  @Test
  public void  canRunExposureByRegionWithThresholdConstructionFunction() throws Exception {
    runCommand.modelId = "exposure-by-region";
    runCommand.parameters = Arrays.asList("analysis.function=threshold_by_construction");
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","Region","Exposed_buildings");

    assertThat(results, contains(
        Arrays.asList("","10"),
        Arrays.asList("Aleipata Itupa i Lalo","476"),
        Arrays.asList("Aleipata Itupa i Luga","307"),
        Arrays.asList("Falealili","679"),
        Arrays.asList("Lepa","262"),
        Arrays.asList("Lotofaga","126")
    ));
  }

  @Test
  public void canGroupByConsequencewithExposureLevelFunction() throws Exception {
    runCommand.modelId = "group-by-consequence";
    runCommand.parameters = Arrays.asList("analysis.function=exposure_level");
    runCommand.doCommand(project);

    List<List<String>> results = openCsv("event-impact.csv","consequence.exposed","consequence.level",
        "Total_buildings");

    assertThat(results, contains(
        Arrays.asList("0","N/A", "4201"),
        Arrays.asList("1","Exposure >0.0m to <=1.0m","473"),
        Arrays.asList("1","Exposure >1.0m to <=2.0m","394"),
        Arrays.asList("1","Exposure >2.0m to <=3.0m","472"),
        Arrays.asList("1","Exposure >3.0m","720")
    ));
  }

  @Before
  public void populateProjectAndSetupCommandsFromHome() {
    populateProjectAndSetupCommands(stdhome().resolve("project-complete.ini"));
  }
}
