/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.cli.tests;
import static nz.org.riskscape.engine.FileSystemMatchers.*;
import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.io.File;
import java.nio.file.Path;
import java.util.List;
import java.util.Set;

import org.junit.Test;

import nz.org.riskscape.engine.Assert;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.cli.BookmarksCommand;
import nz.org.riskscape.engine.cli.ExitException;
import nz.org.riskscape.engine.data.BookmarkProblems;
import nz.org.riskscape.engine.data.InputDataProblems;
import nz.org.riskscape.engine.problem.ProblemMatchers;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;

public class BookmarksTest extends BaseModelRunCommandTest {

  BookmarksCommand.Eval bookmarkEvalCommand;

  @Override
  public ResultOrProblems<Project> populateProjectAndSetupCommands(Path home) {
    return super.populateProjectAndSetupCommands(home).map(p -> {
      bookmarkEvalCommand = this.setupCommand(new BookmarksCommand.Eval());
      p.setOutputBaseLocation(getTempDirectory().toUri());
      return p;
    });
  }

  @Test
  public void canAdjustInputAttributesUsingSetAttribute() throws Exception {
    setCommandInput("input('rearranged')");
    evalCommand.pipelineFile= "-";
    assertNull(evalCommand.run());

    Set<List<String>> rows = openCsvUniqueData("input.csv", "first_name", "last_name", "age", "height", "name");

    assertThat(
      rows,
      containsInAnyOrder(
        contains("Peter","Venkman", "33", "188cm", "Peter Venkman"),
        contains("The","Slimer", "543", "79cm", "The Slimer"),
        contains("Jean","Genie", "77", "134cmcm", "Jean Genie"),
        contains("Gary","Good-Guy", "58", "169cm", "Gary Good-Guy")
      )
    );
  }

  @Test
  public void anEvalExceptionFromSetAttributeExpressionWillKillTheModel() throws Exception {
    setCommandInput("input('bad-data')");
    evalCommand.pipelineFile= "-";
    ExitException ex = Assert.assertThrows(ExitException.class, () -> evalCommand.run());
    assertThat(ex.getProblem(), ProblemMatchers.problemsInTree(equalTo(InputDataProblems.get().invalidTupleError())));
  }

  @Test
  public void anEvalExceptionFromSetAttributeWillBeSwallowedIfSkipInvalidSet() throws Exception {
    setCommandInput("input('bad-data-skipped')");
    evalCommand.pipelineFile= "-";
    assertNull(evalCommand.run());

    Set<List<String>> rows = openCsvUniqueData("input.csv", "first_name", "last_name", "age", "height", "height_mm");

    assertThat(
      rows,
      containsInAnyOrder(
        contains("Peter","Venkman", "33", "188", "1880mm"),
        contains("The","Slimer", "543", "79", "790mm"),
        contains("Gary","Good-Guy", "58", "169", "1690mm")
      )
    );
  }

  @Test
  public void canEvaluateABookmark(){
    bookmarkEvalCommand.bookmarkId = "points-example";
    bookmarkEvalCommand.run();
    Path outputDir = getTempDirectory().resolve("bookmark-eval");
    assertThat(outputDir, hasFile(fileWithName("points-example.shp")));
  }

  @Test
  public void canEvaluateABookmarkToADifferentFileFormat() {
    bookmarkEvalCommand.bookmarkId = "points-example";
    bookmarkEvalCommand.format = "geopackage";
    bookmarkEvalCommand.run();
    Path outputDir = getTempDirectory().resolve("bookmark-eval");
    assertThat(outputDir, hasFile(fileWithName("points-example.gpkg")));
  }

  @Test
  public void getSensibleErrorWithBadBookmarkName() {
    bookmarkEvalCommand.bookmarkId = "points_badname";
    ExitException ex = Assert.assertThrows(ExitException.class, () -> bookmarkEvalCommand.run());

    assertThat(ex.getProblem(), hasAncestorProblem(
            is(BookmarkProblems.get().notBookmarkOrFile("points_badname"))));
  }

  @Test
  public void getSensibleErrorWithLocationDoesntExist() {
    bookmarkEvalCommand.bookmarkId = "points_invalid";
    ExitException ex = Assert.assertThrows(ExitException.class, () -> bookmarkEvalCommand.run());
    File badLocation = stdhome().resolve("does_not_exist.csv").toAbsolutePath().toFile();
    assertThat(ex.getProblem(), hasAncestorProblem(
        is(Problem.error("File '%s' does not exist", badLocation))
    ));
  }

  @Test
  public void canConstructAMapAttributeExpressionWithVariousConstants() throws Exception {
    runCommand.modelId = "map-attribute-constants";

    assertNull(runCommand.run());

    Set<List<String>> rows = openCsvUniqueData("input.csv", "integer", "floating", "boolean", "text");

    assertThat(
      rows,
      containsInAnyOrder(
        contains("1","1.0", "true", "some text")
      )
    );
  }

}
