/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.bld.loc;

import static org.junit.Assert.*;

import org.junit.Test;

import nz.org.riskscape.pipeline.PipelineParser;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration.Found;

public class AtLastChangeTest {

  @Test
  public void canFindLastStepAdded() throws Exception {
    PipelineDeclaration astA = parse("foo -> bar");
    PipelineDeclaration astB = parse("foo -> bar -> baz");
    Found expected = findStep(astB, "baz");
    assertEquals(expected, findLastChange(astA, astB));
  }

  @Test
  public void canFindNewChainAdded() throws Exception {
    PipelineDeclaration astA = parse("foo -> bar");
    PipelineDeclaration astB = parse("foo -> bar", "input()");
    Found expected = findStep(astB, "input");
    assertEquals(expected, findLastChange(astA, astB));
  }

  @Test
  public void canFindNewStepInChain() throws Exception {
    PipelineDeclaration astA = parse("foo -> bar", "input()");
    PipelineDeclaration astB = parse("foo -> bar", "input() -> baz");
    Found expected = findStep(astB, "baz");
    assertEquals(expected, findLastChange(astA, astB));

    // add a new step to the first chain, just to make sure
    PipelineDeclaration astC = parse("foo -> bar -> new", "input() -> baz");
    expected = findStep(astC, "new");
    assertEquals(expected, findLastChange(astB, astC));
  }

  @Test
  public void canFindAmendedStepInChain() throws Exception {
    PipelineDeclaration astA = parse("foo -> bar", "input() -> old -> baz");
    PipelineDeclaration astB = parse("foo -> bar", "input() -> new -> baz");
    Found expected = findStep(astB, "new");
    assertEquals(expected, findLastChange(astA, astB));
  }

  private Found findLastChange(PipelineDeclaration astA, PipelineDeclaration astB) {
    return AtLastChange.INSTANCE.findLastChange(astA, astB);
  }

  private Found findStep(PipelineDeclaration parsed, String withIdent) {
    return parsed.find((chain, step) -> step.getIdent().equals(withIdent)).orElse(null);
  }

  PipelineDeclaration parse(String... source) {
    return PipelineParser.INSTANCE.parsePipeline(String.join("\n", source));
  }
}
