/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.bind;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.Test;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.bind.ParamProblems;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.wizard.Composite;
import nz.org.riskscape.wizard.CompositeField;

public class CompositeBinderTest extends ProjectTest {

  CompositeBinder binder = new CompositeBinder();
  ResultOrProblems<?> result;

  @Composite @EqualsAndHashCode(of={"foo", "bar"})
  public static class Simple {

    public Simple() {
    }

    public Simple(int foo, double bar) {
      this.foo = foo;
      this.bar = bar;
    }

    @CompositeField
    public Integer foo;

    public List<Void> notAComposite;

    @CompositeField
    public Double bar;

  }

  @Composite
  public static class Nested {

    @CompositeField @Getter
    public Simple simple;
  }

  @Test
  public void canBindACompositeClassFromAMap() {
    bind("foo", "1", "bar", "2");
    assertThat(
      result,
      Matchers.result(equalTo(new Simple(1, 2)))
    );
  }

  @Test
  public void aWarningIsGivenIfThereAreSurplusEntries() throws Exception {
    bind("foo", "-1", "bar", "2.1", "baz", "test");

    assertThat(
        result,
        Matchers.resultWithProblems(
            equalTo(new Simple(-1, 2.1)),
            contains(is(ParamProblems.get().ignored("baz")))
        ));
  }

  @Test
  public void failsIfThereAreMissingEntries() throws Exception {
    bind("foo", "-1");

    assertThat(
      result,
      Matchers.failedResult(equalTo(ParamProblems.get().wrongNumberGiven("bar", "1", 0)))
    );
  }

  @Test
  public void includesErrorsFromAllBindingFailures() throws Exception {
    bind("foo", "twelve", "bar", "infinity");

    assertThat(
      result,
      Matchers.failedResult(
        Matchers.equalIgnoringChildren(Problems.foundWith(CompositeField.class, "foo")),
        Matchers.equalIgnoringChildren(Problems.foundWith(CompositeField.class, "bar"))
      )
    );
  }

  @Test
  public void anyWarningsFromOtherBindingsAreIncluded() throws Exception {
    engine.getBinders().add(binder);


    result = binder.bindValue(
      project.newBindingContext(),
      map("simple", map("foo", "1", "bar", "2", "surplus", "junk")),
      Nested.class
    );


    assertThat(
        result,
        Matchers.resultWithProblems(
            hasProperty("simple", equalTo(new Simple(1, 2))),
            contains(
                is(Problems.foundWith(CompositeField.class, "simple", ParamProblems.get().ignored("surplus")))
        )));
  }


  private Map<?, ?> map(Object... keyValues) {
    Map<Object, Object> map = new HashMap<>();

    for (int i = 0; i < keyValues.length; i += 2) {
      Object key = keyValues[i];
      Object value = keyValues[i + 1];
      map.put(key, value);
    }

    return map;
  }


  private Object bind(Object... keyValues) {
    result = binder.bindValue(bindingContext, map(keyValues), Simple.class);
    return result.orElse(null);
  }
}
