/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard;

import java.util.ArrayList;
import java.util.List;

import com.google.common.collect.ImmutableList;

import lombok.Getter;
import nz.org.riskscape.wizard.bld.IncrementalBuildState;
import nz.org.riskscape.wizard.bld.PipelineChange;
import nz.org.riskscape.wizard.bld.change.NoChange;
import nz.org.riskscape.wizard.survey2.Phase;

/**
 * A simple {@link QuestionSet} that work on a static list of {@link Question}s. Suitable for tests that need
 * a {@link Question} that must be part of a set, but not much else.
 */
public class TestQuestionSet extends BaseQuestionSet {

  private final List<Question> list = new ArrayList<>();

  @Getter
  private final Phase phase = Phase.EMPTY_PHASE;

  public TestQuestionSet(String id, Survey survey) {
    super(id, survey);
  }

  public Question add(Question question) {
    question = question.inSet(this);
    list.add(question);
    return question;
  }

  @Override
  public PipelineChange getPipelineChange(IncrementalBuildState buildState, Answer answer) {
    return new NoChange(answer);
  }

  @Override
  public List<Question> getQuestions() {
    return ImmutableList.copyOf(this.list);
  }

  public Question get(String questionName) {
    return list.stream().filter(q -> q.getName().equals(questionName)).findAny().get();
  }

  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder();
    builder.append("TestQuestionSet[");
    builder.append("id=").append(id);
    builder.append(" questions=");
    boolean first = true;
    for (Question question : list) {
      if (!first) {
        builder.append(", ");
      }
      builder.append(question.getName());
      first = false;
    }
    builder.append("]");
    return builder.toString();
  }
}
