/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard;


import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.List;

import org.junit.Test;

import com.google.common.collect.Lists;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.relation.ListRelation;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.types.Enumeration;
import nz.org.riskscape.engine.types.Integer;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Text;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.WithinRange;
import nz.org.riskscape.engine.types.WithinSet;
import nz.org.riskscape.rl.ast.Expression;
import nz.org.riskscape.wizard.ExpressionHelper.AttributeMetadata;
import nz.org.riskscape.wizard.ExpressionHelper.AttributeValue;

public class ExpressionHelperTest {

  Question question = new Question("test", Expression.class);

  Struct nestedStruct = Struct.of("foo", Types.TEXT);
  Struct testContext = Struct.of("name", Nullable.of(Types.TEXT),
      "value", new WithinRange(Types.INTEGER, 1, 10),
      "quality", new WithinSet(Types.TEXT, "poor", "okay", "good", "excellent"))
      .add("material", Enumeration.oneBased("plastic", "wood", "metal"))
      .add("geom", Types.GEOMETRY)
      .add("nested", Nullable.of(nestedStruct));

  @Test
  public void canHelpListAllMemberTypes() {
    ExpressionHelper helper = new ExpressionHelper(question, testContext);

    List<AttributeMetadata> attributes = helper.getAttributeMetadata();
    assertThat(attributes, contains(
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("name"))),
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("value"))),
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("quality"))),
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("material"))),
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("geom"))),
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("nested"), nestedStruct.getEntry("foo"))),
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("nested")))
    ));
  }

  @Test
  public void canHelpWithDesiredMemberTypes() {
    ExpressionHelper helper = new ExpressionHelper(question, testContext);

    List<AttributeMetadata> attributes = helper.getAttributeMetadata(Lists.newArrayList(
        Text.class, Integer.class, Enumeration.class
    ));
    assertThat(attributes, contains(
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("name"))),
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("value"))),   // value has integer backing
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("quality"))), // quality has text backing
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("material"))),
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("nested"), nestedStruct.getEntry("foo")))
    ));
  }

  @Test
  public void canHelpWithTextMembers() {
    ExpressionHelper helper = new ExpressionHelper(question, testContext);

    List<AttributeMetadata> attributes = helper.getAttributeMetadata(Lists.newArrayList(
        Text.class
    ));
    assertThat(attributes, contains(
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("name"))),
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("quality"))), // quality has text backing
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("nested"), nestedStruct.getEntry("foo")))
    ));

    // cannot get values from a text field unless the helper has the relation
    assertThat(helper.getValues(attributes.get(0)), is(empty()));

    // we can get possible values from a set
    assertThat(helper.getValues(attributes.get(1)), contains(
        new AttributeValue("excellent"),
        new AttributeValue("good"),
        new AttributeValue("okay"),
        new AttributeValue("poor")
    ));
  }

  @Test
  public void canHelpWithIntegerMembers() {
    ExpressionHelper helper = new ExpressionHelper(question, testContext);

    List<AttributeMetadata> attributes = helper.getAttributeMetadata(Lists.newArrayList(
        Integer.class
    ));
    assertThat(attributes, contains(
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("value")))
    ));

    // an integer range returns empty as well. Maybe it could have values for integer backing type
    // but that approach wouldn't scale well to a float backing type
    assertThat(helper.getValues(attributes.get(0)), is(empty()));
  }

  @Test
  public void canHelpWithEnumerations() {
    ExpressionHelper helper = new ExpressionHelper(question, testContext);

    List<AttributeMetadata> attributes = helper.getAttributeMetadata(Lists.newArrayList(
        Enumeration.class
    ));
    assertThat(attributes, contains(
        new AttributeMetadata(Lists.newArrayList(testContext.getEntry("material")))

    ));

    // we can get possible values from an enumeration
    assertThat(helper.getValues(attributes.get(0)), contains(
        // enumerations are listed in alphabetical order rather than definition order
        // (we use the string value so the order doesn't really matter)
        new AttributeValue("metal", "metal"),
        new AttributeValue("plastic", "plastic"),
        new AttributeValue("wood", "wood")
    ));

  }

  @Test
  public void canGetActualValuesFromRelations() {
    Struct relationType = Struct.of("name", Types.TEXT, "value", Types.INTEGER);
    Relation testRelation = new ListRelation(relationType,
        Tuple.ofValues(relationType, "foo", 10L),
        Tuple.ofValues(relationType, "bar", 12L),
        Tuple.ofValues(relationType, "bar", 10L),
        Tuple.ofValues(relationType, "foo", 8L)
    );

    ExpressionHelper helper = new ExpressionHelper(question, testRelation, "relation");
    List<AttributeMetadata> attributes = helper.getAttributeMetadata(Lists.newArrayList(
        Integer.class, Text.class
    ));

    assertThat(helper.getValues(attributes.get(0)), contains(
        // text is sorted by string value
        new AttributeValue("bar"),
        new AttributeValue("foo")
    ));

    assertThat(helper.getValues(attributes.get(1)), contains(
        // numbers are sorted by value
        new AttributeValue(8L),
        new AttributeValue(10L),
        new AttributeValue(12L)
    ));

  }

}
