/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;

import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.problem.Problem.Severity;

public class AnswerTest {

  Question question = new Question("foo", Object.class).optionalMany();
  BindingContext context = Mockito.mock(BindingContext.class);

  Map<Object, ResultOrProblems<?>> bindingResults = new HashMap<>();

  Problem warning = GeneralProblems.get().deprecated("this", "that").withSeverity(Severity.WARNING);
  Problem error = GeneralProblems.get().required("lack of respect");

  @Before
  public void setup() {
    when(context.bind(Mockito.any(String.class), Mockito.eq(Object.class))).thenAnswer(inv -> {
      String valueToBind = inv.getArgument(0);

      return bindingResults.getOrDefault(valueToBind, ResultOrProblems.of(valueToBind));
    });
  }

  @Test
  public void bindAllReturnsAllTheBoundValuesInAnAnswer() throws Exception {
    assertThat(
      Answer.bindAll(context, question, Arrays.asList("that", "was", "great")),
      equalTo(ResultOrProblems.of(Answer.strings(question, "that", "was", "great")))
    );
  }

  @Test
  public void bindAllIncludesAWarning() throws Exception {
    // one by itself
    bindingResults.put("excellent", ResultOrProblems.of("adequate", warning));

    assertThat(
      Answer.bindAll(context, question, Arrays.asList("excellent")),
      equalTo(ResultOrProblems.of(Answer.single(question, "adequate", "excellent"), warning))
    );
  }

  @Test
  public void bindAllIncludesAnyWarningsFromAnyIndividualBindings() throws Exception {
    // in the context of others - note I don't bother returning a different bound value - too much leg work with the
    // answer constructor
    bindingResults.put("excellent", ResultOrProblems.of("excellent", warning));

    assertThat(
      Answer.bindAll(context, question, Arrays.asList("that", "was", "excellent")),
      equalTo(ResultOrProblems.of(Answer.strings(question, "that", "was", "excellent"), warning))
    );
  }

  @Test
  public void bindAllReturnsAFailedResultIfAnyBindingsFail() throws Exception {
    bindingResults.put("respect", ResultOrProblems.failed(error));

    assertThat(
      Answer.bindAll(context, question, Arrays.asList("respect")),
      equalTo(ResultOrProblems.failed(error))
    );
  }


}
