/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeDiagnosingMatcher;

import nz.org.riskscape.engine.MatcherCollection;

public class AnswerMatchers implements MatcherCollection {

  /**
   * @return a Matcher that checks the answer is from a question with a given name and has an unbound value matching
   * the given `unboundMatcher`, e.g. `answerHasOriginalResponse("foo-question" equalTo("some-user-input"))`
   */
  public static Matcher<Answer> answerHasOriginalResponse(String questionName, Matcher<?> unboundMatcher) {
    return new TypeSafeDiagnosingMatcher<Answer>(Answer.class) {

      @Override
      public void describeTo(Description description) {
        description
          .appendText("answer to question named ")
          .appendValue(questionName)
          .appendText(" with unbound ")
          .appendDescriptionOf(unboundMatcher);
      }

      @Override
      protected boolean matchesSafely(Answer item, Description mismatchDescription) {
        boolean matches = true;

        if (!item.getQuestion().getName().equals(questionName)) {
          matches = false;
          mismatchDescription.appendText("question name was ").appendValue(item.getQuestion().getName());
        }

        if (item.getOriginalResponses().size() != 1) {
          matches = false;
          mismatchDescription.appendText("expected a response, but there were none ");
        } else {
          if (!unboundMatcher.matches(item.getResponse().getOriginalInput())) {
            matches = false;
            mismatchDescription.appendText(" unbound value was ");
            unboundMatcher.describeMismatch(item.getResponse().getOriginalInput(), mismatchDescription);
          }
        }

        return matches;
      }
    };
  }

  /**
   * @return a Matcher that checks the answer is from a question with a given name and has a bound value matching
   * the given `boundMatcher`, e.g. `answerHasBoundValue("foo-question" equalTo(someBoundValue))`
   */
  public static Matcher<Answer> answerHasBoundValue(String questionName, Matcher<?> boundMatcher) {
    return new TypeSafeDiagnosingMatcher<Answer>(Answer.class) {

      @Override
      public void describeTo(Description description) {
        description
          .appendText("answer to question named ")
          .appendValue(questionName)
          .appendText(" with bound value")
          .appendDescriptionOf(boundMatcher);
      }

      @Override
      protected boolean matchesSafely(Answer item, Description mismatchDescription) {
        boolean matches = true;

        if (!item.getQuestion().getName().equals(questionName)) {
          matches = false;
          mismatchDescription.appendText("question name was ").appendValue(item.getQuestion().getName());
        }

        if (item.getOriginalResponses().size() != 1) {
          matches = false;
          mismatchDescription.appendText("expected a response, but there were none ");
        } else {
          if (!boundMatcher.matches(item.getResponse().getBoundValue())) {
            matches = false;
            mismatchDescription.appendText(" bound value was ");
            boundMatcher.describeMismatch(item.getResponse().getBoundValue(), mismatchDescription);
          }
        }

        return matches;
      }
    };
  }

  /**
   * @return a Matcher that checks the answer is from a question with a given name that the question was
   * skipped
   */
  public static Matcher<Answer> skippedQuestion(String questionName) {
    return new TypeSafeDiagnosingMatcher<Answer>(Answer.class) {

      @Override
      public void describeTo(Description description) {
        description
          .appendText("skipped question named ")
          .appendValue(questionName);
      }

      @Override
      protected boolean matchesSafely(Answer item, Description mismatchDescription) {
        boolean matches = true;

        if (!item.getQuestion().getName().equals(questionName)) {
          matches = false;
          mismatchDescription.appendText(" question name was ").appendValue(item.getQuestion().getName());
        }

        if (item.getOriginalResponses().size() != 0) {
          matches = false;
          mismatchDescription.appendText(" not skipped");
        }

        return matches;
      }
    };
  }

}
