/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape;

import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasProperty;
import static org.hamcrest.Matchers.is;
import static org.junit.Assert.assertThat;

import java.util.Locale;

import org.junit.Test;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.TypedProperty;
import nz.org.riskscape.engine.data.ResolvedBookmark;
import nz.org.riskscape.engine.i18n.MutableMessageSource;
import nz.org.riskscape.engine.sort.SortBy;
import nz.org.riskscape.wizard.EmptySurvey;
import nz.org.riskscape.wizard.Question;
import nz.org.riskscape.wizard.QuestionSet;
import nz.org.riskscape.wizard.Survey;
import nz.org.riskscape.wizard.TestQuestionSet;
import nz.org.riskscape.wizard.WizardModelParameter;
import nz.org.riskscape.wizard.bld.IncrementalBuildState;
import nz.org.riskscape.wizard.model2.report.SortExpression;

public class WizardModelParameterTest extends ProjectTest {

  MutableMessageSource messageSource = new MutableMessageSource();
  Survey survey = new EmptySurvey("foo", messageSource);
  IncrementalBuildState buildState = IncrementalBuildState.empty(
    new IncrementalBuildState.Context(survey, executionContext)
  );
  QuestionSet questionSet = new TestQuestionSet("test-question-set", survey);


  @Test
  public void resolvedBookmarkQuestionHasBookmarkProperty() {
    Question question = new Question("bookmark-question", ResolvedBookmark.class)
      .inSet(questionSet);

    WizardModelParameter param = new WizardModelParameter(question, buildState);

    assertThat(param.getProperties(), contains(TypedProperty.BOOKMARK));
  }

  enum TestEnum {
    ONE,
    TWO,
    THREE
  }

  @Test
  public void enumQuestionGetChoicesReturnsCorrectChoices() {
    Question question  = new Question("enum-question", TestEnum.class)
      .inSet(questionSet);

    WizardModelParameter param = new WizardModelParameter(question, buildState);

    assertThat(param.getChoices(Locale.getDefault()), contains(
      hasProperty("value", is(TestEnum.ONE.name())),
      hasProperty("value", is(TestEnum.TWO.name())),
      hasProperty("value", is(TestEnum.THREE.name()))
    ));
  }

  @Test
  public void parameterHasI18n() {
    Locale french = Locale.FRENCH;

    String defaultTitle = "A title";
    String defaultDescription = "A super cool description";
    messageSource.addMessage("question.test-question-set.test-question.title", defaultTitle);
    messageSource.addMessage("question.test-question-set.test-question.description", defaultDescription);

    String frenchTitle = "Un titre";
    String frenchDescription = "Une description super cool";
    messageSource.addMessage(french, "question.test-question-set.test-question.title", frenchTitle);
    messageSource.addMessage(french, "question.test-question-set.test-question.description", frenchDescription);

    Question question  = new Question("test-question", TestEnum.class)
      .inSet(questionSet);

    WizardModelParameter param = new WizardModelParameter(question, buildState);


    assertThat(param.getDescription(Locale.getDefault()), equalTo(defaultTitle + " " + defaultDescription));
    assertThat(param.getDescription(french), equalTo(frenchTitle + " " + frenchDescription));
  }

  @Test
  public void nameTurnsIntoNiceLabelByDefault() {

    Parameter parameter = Parameter.required("report-event-impact.sort-by[0][direction]", SortBy.Direction.class);

    Question question  = new Question("test-question", SortExpression.class)
      .inSet(questionSet);

    WizardModelParameter modelParam = new WizardModelParameter(parameter, question, buildState);
    assertThat(modelParam.getLabel(), equalTo("Report event impact sort by 0 direction"));
  }
}
