/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.survey2;

import java.util.Optional;

import javax.annotation.Nullable;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.ToString;
import nz.org.riskscape.engine.i18n.TranslationContext;
import nz.org.riskscape.wizard.Choice;
import nz.org.riskscape.wizard.QuestionSet;
import nz.org.riskscape.wizard.Survey;

/**
 * Special parameter type used by surveys that is used to indicate a choice of question set.  We can't use a QuestionSet
 * directly as there's no way to bind them without interrogating the {@link Survey}, which complicates the binding
 * (and/or management of QuestionSets).  So instead, we have a PickQuestionSet which gives us a special parameter type
 * that we can look for in our survey logic and a pointer to the question set that was chosen.
 *
 * The class also has a field for setting the questionSet which can't be relied on as being there, but will be set in
 * specific circumstances.  It's a bit of a kludge but makes it nice and easy for passing it around and/or memoizing the
 * question set.
 */
@EqualsAndHashCode(of = {"questionSetId"}) @ToString(of = "questionSetId")
public class PickQuestionSet {

  public PickQuestionSet() {}

  public PickQuestionSet(String questionSetId) {
    this.questionSetId = questionSetId;
  }

  public PickQuestionSet(QuestionSet qs) {
    this.questionSetId = qs.getId();
    this.questionSet = qs;
  }

  @Getter
  public String questionSetId;

  @Nullable
  public QuestionSet questionSet;

  /**
   * @return true if the user's choice was to skip the question sets that were offered to them.
   */
  public boolean isSkip() {
    return QuestionSet.SKIP.equals(questionSet);
  }

  /**
   * @return the QuestionSet as a Choice that can be presented in either the CLI or UI
   */
  public Choice asChoice() {
    return new Choice(questionSet.getId(), questionSet.getName(), Optional.empty(), this) {

      public String getDescription(TranslationContext context) {
        return questionSet.getDescription(context.getLocale()).orElse("");
      }

      public String getLabel(TranslationContext context) {
        return questionSet.getTitle(context.getLocale()).orElse(questionSet.getName());
      }
    };
  }
}
