/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.model2;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.rl.ast.Expression;
import nz.org.riskscape.rl.ast.StructDeclaration;

/**
 * Provides some convenience methods to assist generating pipeline DSL code.
 */
public class DslHelper {

  /**
   * Removes the '{' '}'s from a struct expression with a single member. This
   * avoids double-wrapping it in the pipeline DSL and thus getting weird inferred
   * identifier names.
   */
  private static String unwrapStructExpression(Expression expr) {
    Optional<StructDeclaration> struct = expr.isA(StructDeclaration.class);
    if (struct.isPresent()) {
      List<StructDeclaration.Member> members = struct.get().getMembers();
      if (members.size() == 1) {
        // note that the format might change slightly here from what the user input,
        // e.g. "foo as bar" becomes "bar: foo", but as this is going into a larger
        // struct declaration (like a select step) it shouldn't matter
        return members.get(0).toSource();
      }
    }
    // this is not a struct, or it has multiple members, so just display it as is
    return expr.toSource();
  }

  /**
   * Combines a list of expressions into a comma-separated struct declaration
   * string, minus the {}s.
   */
  public static String asStructMembers(List<? extends Expression> responses) {
    return responses.stream()
        .map(expr -> unwrapStructExpression(expr))
        .collect(Collectors.joining(", "));
  }

  /**
   * @return the Type of the specified attribute (if present) within the parent struct.
   */
  public static <T extends Type> Optional<T> getAttributeType(Struct parent, String attributeName, Class<T> type) {
    return parent.getMember(attributeName)
        .flatMap(member -> member.getType().findAllowNull(type));
  }
}
