/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.bld.loc;

import java.util.Collections;
import java.util.List;
import java.util.Optional;

import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration.Found;
import nz.org.riskscape.pipeline.ast.StepChain;
import nz.org.riskscape.pipeline.ast.StepReference;
import nz.org.riskscape.wizard.bld.IncrementalBuildState;
import nz.org.riskscape.wizard.bld.PipelineChange;

/**
 * Can identify a location at which a {@link PipelineChange} can apply to the build state of a pipeline during wizard
 * process.
 *
 * A change should typically be based on a description of the location and be as independent of the structure of the
 * pipeline as possible to allow new questions and changes to be dropped in as non-invasively as possible.  For example,
 * a change that says 'at the end of the chain that contains such and such data' vs 'find step named y'
 */
public interface ChangeLocation {

  /**
   * Magic value for null
   */
  Found EMPTY = new Found(PipelineDeclaration.EMPTY, StepChain.EMPTY, new StepReference("null"));

  /**
   * Magic locator that signals that a change applies at the very beginning of the pipeline, which might vary in its
   * interpretation, depending on the change it is applied to, but typically only meaningful for a new StepChain being
   * inserted
   */
  ChangeLocation START = buildState -> Collections.singletonList(EMPTY);

  /**
   * Magic locator that signals that a change applies at the very end of the pipeline, which might vary in its
   * interpretation, depending on the change it is applied to, but typically only meaningful for a new StepChain being
   * inserted
   */
  ChangeLocation END = buildState -> Collections.singletonList(EMPTY);

  /**
   * @return true if this change location expects to return more than one {@link Found} location.
   */
  default boolean isMultiple() {
    return false;
  }

  /**
   * @return a list of locations where the change should be made to the given state.
   */
  List<Found> find(IncrementalBuildState buildState);

  default Optional<Found> findOne(IncrementalBuildState buildState) {
    List<Found> results = find(buildState);

    if (results.size() == 1) {
      return Optional.of(results.get(0));
    } else {
      return Optional.empty();
    }
  }

  @Override
  String toString();

  /**
   * @return a ChangeLocation centred on a step definition with the given name
   */
  static ChangeLocation atStep(String stepDefinitionName) {
    return new AtStepNamed(stepDefinitionName);
  }
}
