/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.bld.loc;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

import nz.org.riskscape.pipeline.ast.PipelineDeclaration;
import nz.org.riskscape.pipeline.ast.PipelineDeclaration.Found;
import nz.org.riskscape.pipeline.ast.StepChain;
import nz.org.riskscape.pipeline.ast.StepDeclaration;
import nz.org.riskscape.wizard.bld.IncrementalBuildState;

/**
 * Identifies a location to apply a change by finding the last step that changed in the build state.  Finds the change
 * in reverse order, so that the last step and chain among all changes are returned.
 */
public class AtLastChange implements ChangeLocation {

  public static final AtLastChange INSTANCE = new AtLastChange();

  private AtLastChange() {
  }

  @Override
  public List<Found> find(IncrementalBuildState buildState) {
    return findOne(buildState).map(v -> Collections.singletonList(v)).orElse(Collections.emptyList());
  }

  @Override
  public Optional<Found> findOne(IncrementalBuildState buildState) {
    if (buildState.isEmpty()) {
      return Optional.empty();
    }

    // we step back through the build states to find some older AST (i.e. that differs),
    // and then compare the two to work out what has changed
    PipelineDeclaration latestAst = buildState.getAst();
    IncrementalBuildState ptr = buildState;
    PipelineDeclaration oldAst = null;
    while (!ptr.isEmpty()) {
      ptr = ptr.getLast();
      oldAst = ptr.getAst();

      if (!oldAst.equals(latestAst)) {
        return Optional.ofNullable(findLastChange(oldAst, latestAst));
      }
    }

    // there's no older version of the pipeline to compare, so just return the last step-chain
    return Optional.of(Found.last(latestAst, latestAst.getLast()));
  }

  @Override
  public String toString() {
    return "at last change";
  }

  protected Found findLastChange(PipelineDeclaration oldAst, PipelineDeclaration latestAst) {
    if (oldAst.getChains().size() != latestAst.getChains().size()) {
      // simple case - a new chain was added
      return Found.last(latestAst, latestAst.getLast());
    } else {
      // walk backwards until we find the chain that differs/was modified
      Iterator<StepChain> oldChains = revChainIterator(oldAst);
      Iterator<StepChain> latestChains = revChainIterator(latestAst);

      while (oldChains.hasNext()) {
        StepChain oldChain = oldChains.next();
        StepChain latestChain = latestChains.next();

        if (!oldChain.equals(latestChain)) {
          // now find the step that was changed
          return findLatestChange(latestAst, oldChain, latestChain);
        }
      }

      throw new AssertionError("chains are all equal yet asts are not");
    }
  }

  private Found findLatestChange(PipelineDeclaration ast, StepChain oldChain, StepChain latestChain) {
    if (oldChain.size() != latestChain.size()) {
      // simple case - the latest chain has had new steps added
      return Found.last(ast, latestChain);
    } else {
      // a step must've been modified somewhere - walk backwards until we find it
      Iterator<StepDeclaration> oldSteps = revStepIterator(oldChain);
      Iterator<StepDeclaration> latestSteps = revStepIterator(latestChain);

      while (oldSteps.hasNext()) {
        StepDeclaration oldStep = oldSteps.next();
        StepDeclaration latestStep = latestSteps.next();

        if (!oldStep.equals(latestStep)) {
          return new Found(ast, latestChain, latestStep);
        }
      }

      throw new AssertionError("steps are all equal yet chains are not");
    }
  }

  private Iterator<StepDeclaration> revStepIterator(StepChain chain) {
    return revIterator(chain.getSteps());
  }

  private Iterator<StepChain> revChainIterator(PipelineDeclaration pipeline) {
    return revIterator(pipeline.getChains());
  }

  private <T> Iterator<T> revIterator(List<T> list) {
    List<T> clone = new ArrayList<>(list);
    Collections.reverse(clone);
    return clone.iterator();
  }

}
