/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.bld.dsl;

import java.util.List;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.wizard.Answer;
import nz.org.riskscape.wizard.Question;
import nz.org.riskscape.wizard.bld.IncrementalBuildState;
import nz.org.riskscape.wizard.bld.PipelineChange;
import nz.org.riskscape.wizard.bld.loc.ChangeLocation;

/**
 * Collected arguments that are given to a {@link PipelineChanger} so that it can return a {@link PipelineChange}
  */
@RequiredArgsConstructor
public class PipelineChangeInput {

  /**
   * The build point that was triggered
   */
  @Getter
  private final BuildPoint buildPoint;

  @Getter
  private final Answer answer;

  @Getter
  private final IncrementalBuildState buildState;

  /**
   * Assuming the change was triggered by a single answer, this returns the single bound response that was part of the
   * answer
   */
  public <T> T getResponse(Class<T> expectedType) {
    return answer.getValueAs(expectedType);
  }

  /**
   * Same as {@link #getResponse(Class)} but for Questions that accept multiple responses.
   */
  public <T> List<T> getResponses(Class<T> expectedType) {
    return answer.getValuesAs(expectedType);
  }

  /**
   * @return a ChangeLocation that can be used to find the part of the pipeline that could be changed as part of this
   * question's response.  The resulting location can be used to get the input struct to provide content assist, as well
   * as allowing the change code itself to have a place to make changes.
   */
  public ChangeLocation getChangeLocation() {
    return getQuestion().getChangeLocation();
  }

  public Question getQuestion() {
    return answer.getQuestion();
  }

  /**
   * @return A Struct representing the input scope currently available.
   */
  public Struct getStruct() {
    return buildState.getInputStruct(getQuestion());
  }
}
