/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.wizard.bld.change;

import java.util.Collections;
import java.util.List;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.wizard.Answer;
import nz.org.riskscape.wizard.Question;
import nz.org.riskscape.wizard.QuestionSet;
import nz.org.riskscape.wizard.bld.IncrementalBuildState;
import nz.org.riskscape.wizard.bld.PipelineChange;

/**
 * An answer was given, but no change is to be made to the pipeline's structure
 */
@RequiredArgsConstructor
public class NoChange implements PipelineChange {

  /**
   * @return a handy change that represents an optional question being skipped.  These have to be recorded to allow
   * the {@link QuestionSet#getNextQuestions(IncrementalBuildState)} routine to record the user's option to not answer
   * some questions.
   */
  public static IncrementalBuildState skip(
      IncrementalBuildState buildState,
      List<Question> toSkip
  ) {
    for (Question question : toSkip) {

      if (question.isRequired()) {
        throw new IllegalStateException("you can not skip required questions");
      }

      PipelineChange change =
          new NoChange(new Answer(question, Collections.emptyList()));

      buildState = change.make(buildState).get();
    }

    return buildState;
  }

  @Getter
  private final Answer answer;

  @Override
  public ResultOrProblems<IncrementalBuildState> make(IncrementalBuildState currentState)
      throws BadPipelineChangeException {

    // prevent needless elements on the stack, just in case
    if (currentState.getCreatedBy().equals(this)) {
      return ResultOrProblems.of(currentState);
    } else {
      return ResultOrProblems.of(currentState.append(this, currentState.getAst(), currentState.getRealizedPipeline()));
    }
  }

  @Override
  public String toString() {
    return BasePipelineChange.toString(getClass(), answer, null);
  }

}
